const Loki = require('lokijs');

console.log("🔍 Comprehensive Category and Subcategory Verification...\n");

const db = new Loki("users.db", {
  autoload: true,
  autoloadCallback: function () {
    const categories = db.getCollection("categories");
    const subcategories = db.getCollection("subcategories");
    
    if (!categories || !subcategories) {
      console.log("❌ Collections not found.");
      process.exit(1);
    }

    console.log("📊 DATABASE SUMMARY:");
    console.log(`• Total Categories: ${categories.data.length}`);
    console.log(`• Total Subcategories: ${subcategories.data.length}\n`);

    // Check category status
    const activeCategories = categories.find({ isActive: true });
    const inactiveCategories = categories.find({ isActive: { "$ne": true } });
    
    console.log("📈 CATEGORY STATUS:");
    console.log(`• Active Categories: ${activeCategories.length}`);
    console.log(`• Inactive Categories: ${inactiveCategories.length}\n`);

    // Show inactive categories if any
    if (inactiveCategories.length > 0) {
      console.log("❌ INACTIVE CATEGORIES:");
      inactiveCategories.forEach(cat => {
        console.log(`• ${cat.name} (isActive: ${cat.isActive})`);
      });
      console.log();
    }

    // Sort categories by sortOrder
    const sortedCategories = categories.data.sort((a, b) => {
      if (a.sortOrder !== b.sortOrder) {
        return a.sortOrder - b.sortOrder;
      }
      return a.name.localeCompare(b.name);
    });

    console.log("📋 CATEGORY-WISE SUBCATEGORY BREAKDOWN:\n");
    
    let totalSubcategoriesFound = 0;
    let categoriesWithoutSubcategories = [];
    
    sortedCategories.forEach((category, index) => {
      const categorySubcategories = subcategories.find({ 
        categoryId: category.$loki.toString() 
      });
      
      totalSubcategoriesFound += categorySubcategories.length;
      
      const statusIcon = category.isActive ? "✅" : "❌";
      const subcategoryCount = categorySubcategories.length;
      
      console.log(`${index + 1}. ${statusIcon} ${category.icon} ${category.name}`);
      console.log(`   Status: ${category.isActive ? "Active" : "Inactive"} | Sort Order: ${category.sortOrder}`);
      console.log(`   Subcategories: ${subcategoryCount}`);
      
      if (subcategoryCount === 0) {
        categoriesWithoutSubcategories.push(category.name);
        console.log("   ⚠️  No subcategories found!");
      } else {
        categorySubcategories.forEach(sub => {
          const subStatusIcon = sub.isActive ? "✅" : "❌";
          console.log(`   - ${subStatusIcon} ${sub.icon} ${sub.name} (${sub.slug})`);
        });
      }
      console.log();
    });

    // Summary statistics
    console.log("📊 VERIFICATION SUMMARY:");
    console.log(`• Categories processed: ${sortedCategories.length}`);
    console.log(`• Subcategories found: ${totalSubcategoriesFound}`);
    console.log(`• Expected subcategories: ${subcategories.data.length}`);
    console.log(`• Categories without subcategories: ${categoriesWithoutSubcategories.length}`);
    
    if (categoriesWithoutSubcategories.length > 0) {
      console.log("\n⚠️  CATEGORIES WITHOUT SUBCATEGORIES:");
      categoriesWithoutSubcategories.forEach(catName => {
        console.log(`• ${catName}`);
      });
    }

    // Check for orphaned subcategories
    const orphanedSubcategories = subcategories.data.filter(sub => {
      const parentCategory = categories.findOne({ $loki: parseInt(sub.categoryId) });
      return !parentCategory;
    });

    if (orphanedSubcategories.length > 0) {
      console.log(`\n⚠️  ORPHANED SUBCATEGORIES (${orphanedSubcategories.length}):`);
      orphanedSubcategories.forEach(sub => {
        console.log(`• ${sub.name} (CategoryID: ${sub.categoryId}) - Parent not found`);
      });
    }

    // Active/Inactive subcategory count
    const activeSubcategories = subcategories.find({ isActive: true });
    const inactiveSubcategories = subcategories.find({ isActive: { "$ne": true } });
    
    console.log(`\n📈 SUBCATEGORY STATUS:`);
    console.log(`• Active Subcategories: ${activeSubcategories.length}`);
    console.log(`• Inactive Subcategories: ${inactiveSubcategories.length}`);

    // Final status
    console.log("\n🎯 OVERALL STATUS:");
    const allCategoriesActive = inactiveCategories.length === 0;
    const allSubcategoriesActive = inactiveSubcategories.length === 0;
    const noOrphanedSubcategories = orphanedSubcategories.length === 0;
    
    console.log(`• All categories active: ${allCategoriesActive ? "✅ YES" : "❌ NO"}`);
    console.log(`• All subcategories active: ${allSubcategoriesActive ? "✅ YES" : "❌ NO"}`);
    console.log(`• No orphaned subcategories: ${noOrphanedSubcategories ? "✅ YES" : "❌ NO"}`);
    console.log(`• Database integrity: ${allCategoriesActive && allSubcategoriesActive && noOrphanedSubcategories ? "✅ GOOD" : "⚠️  NEEDS ATTENTION"}`);

    process.exit(0);
  },
});