const Loki = require('lokijs');

console.log("🔄 Updating subcategories structure...");

const db = new Loki("users.db", {
  autoload: true,
  autoloadCallback: function () {
    const categories = db.getCollection("categories");
    const subcategories = db.getCollection("subcategories");
    
    if (!categories || !subcategories) {
      console.log("Collections not found.");
      process.exit(1);
    }

    console.log(`Found ${subcategories.data.length} subcategories to update`);

    // Update each subcategory with proper structure
    subcategories.data.forEach((subcategory, index) => {
      // Find the parent category
      const parentCategory = categories.findOne({ $loki: subcategory.categoryId });
      
      if (parentCategory) {
        // Update subcategory with missing fields
        subcategory.id = `SUBCAT-${Date.now()}-${index.toString().padStart(3, '0')}`;
        subcategory.description = subcategory.description || `${subcategory.name} products and services`;
        subcategory.slug = subcategory.slug || subcategory.name.toLowerCase()
          .replace(/[^a-z0-9 -]/g, '')
          .replace(/\s+/g, '-')
          .replace(/-+/g, '-')
          .trim();
        subcategory.icon = subcategory.icon || '📁';
        subcategory.isActive = subcategory.isActive !== undefined ? subcategory.isActive : true;
        subcategory.sortOrder = subcategory.sortOrder || 0;
        subcategory.createdAt = subcategory.createdAt || new Date().toISOString();
        subcategory.updatedAt = new Date().toISOString();
        
        // Convert categoryId to string (the $loki ID as string)
        subcategory.categoryId = parentCategory.$loki.toString();
        
        subcategories.update(subcategory);
      }
    });

    db.saveDatabase(() => {
      console.log("✅ All subcategories updated successfully!");
      console.log(`Updated ${subcategories.data.length} subcategories`);
      
      // Show sample updated subcategory
      console.log("\nSample updated subcategory:");
      console.log(JSON.stringify(subcategories.data[0], null, 2));
      
      process.exit(0);
    });
  },
});