const fetch = require('node-fetch');

async function testCategoryFilter() {
    console.log('🧪 Testing category filter API...\n');
    
    try {
        // Test categories API
        console.log('1. Testing /api/categories endpoint:');
        const categoriesResponse = await fetch('http://localhost:3001/api/categories');
        const categoriesData = await categoriesResponse.json();
        
        if (categoriesResponse.ok && categoriesData.categories) {
            console.log(`   ✅ Found ${categoriesData.categories.length} categories`);
            const firstCategory = categoriesData.categories[0];
            console.log(`   📂 First category: ${firstCategory.name} (ID: ${firstCategory.id}, $loki: ${firstCategory.$loki})`);
        } else {
            console.log('   ❌ Failed to fetch categories:', categoriesData);
            return;
        }
        
        // Test subcategories API
        console.log('\n2. Testing /api/subcategories endpoint:');
        const subcategoriesResponse = await fetch('http://localhost:3001/api/subcategories');
        const subcategoriesData = await subcategoriesResponse.json();
        
        if (subcategoriesResponse.ok && subcategoriesData.subcategories) {
            console.log(`   ✅ Found ${subcategoriesData.subcategories.length} subcategories`);
            
            // Test filtering logic with first category
            const testCategoryId = categoriesData.categories[0].id; // This should be a string
            const filteredSubs = subcategoriesData.subcategories.filter(sub => 
                sub.categoryId === testCategoryId
            );
            
            console.log(`\n3. Testing filter logic:`);
            console.log(`   🔍 Filtering for categoryId: "${testCategoryId}" (${typeof testCategoryId})`);
            console.log(`   📋 Found ${filteredSubs.length} subcategories`);
            
            if (filteredSubs.length > 0) {
                console.log('   ✅ Filter working! Sample results:');
                filteredSubs.slice(0, 3).forEach(sub => {
                    console.log(`      - ${sub.name} (categoryId: ${sub.categoryId}, parent: ${sub.parentCategory?.name})`);
                });
            } else {
                console.log('   ❌ Filter not working - no results found');
                console.log('   🔧 Debug info:');
                console.log(`      Looking for: "${testCategoryId}"`);
                const uniqueIds = [...new Set(subcategoriesData.subcategories.map(s => s.categoryId))];
                console.log(`      Available categoryIds: ${uniqueIds.join(', ')}`);
            }
            
        } else {
            console.log('   ❌ Failed to fetch subcategories:', subcategoriesData);
        }
        
    } catch (error) {
        console.error('❌ Error testing APIs:', error.message);
        console.log('\n💡 Make sure the development server is running on port 3001');
    }
}

testCategoryFilter();