const http = require('http');
const fs = require('fs');

// Function to check database state
function checkDatabaseState() {
  try {
    const data = JSON.parse(fs.readFileSync('users.db', 'utf8'));
    const orders = data.collections.find(c => c.name === 'orders');
    if (orders && orders.data.length > 0) {
      console.log('\n=== Current Database State ===');
      orders.data.forEach((order, index) => {
        console.log(`${index + 1}. ${order.orderNumber}: ${order.orderStatus} (Updated: ${order.updatedAt || 'N/A'})`);
      });
      return orders.data;
    }
    return [];
  } catch (error) {
    console.error('Error reading database:', error);
    return [];
  }
}

// Function to test API update
function testAPIUpdate(orderNumber, newStatus) {
  return new Promise((resolve, reject) => {
    const updateData = JSON.stringify({ orderStatus: newStatus });

    const options = {
      hostname: 'localhost',
      port: 3001,
      path: `/api/orders/${orderNumber}`,
      method: 'PATCH',
      headers: {
        'Content-Type': 'application/json',
        'Content-Length': Buffer.byteLength(updateData)
      }
    };

    const req = http.request(options, (res) => {
      let data = '';
      res.on('data', (chunk) => { data += chunk; });
      res.on('end', () => {
        console.log(`\n=== API Test Results ===`);
        console.log(`Order: ${orderNumber}`);
        console.log(`New Status: ${newStatus}`);
        console.log(`HTTP Status: ${res.statusCode}`);
        
        if (res.statusCode === 200) {
          try {
            const result = JSON.parse(data);
            console.log('✅ API Response:', result);
            resolve(result);
          } catch (e) {
            console.log('✅ API Response (raw):', data);
            resolve({ success: true });
          }
        } else {
          console.log('❌ API Error:', data);
          reject(new Error(`HTTP ${res.statusCode}: ${data}`));
        }
      });
    });

    req.on('error', (error) => {
      console.error('❌ Request failed:', error.message);
      reject(error);
    });

    req.write(updateData);
    req.end();
  });
}

async function testButtonFunctionality() {
  console.log('🧪 Testing Action Button Functionality...\n');
  
  // Check initial state
  console.log('📊 Initial Database State:');
  const initialOrders = checkDatabaseState();
  
  if (initialOrders.length === 0) {
    console.log('❌ No orders found in database');
    return;
  }
  
  // Find a pending order to test with
  const pendingOrder = initialOrders.find(order => order.orderStatus === 'pending');
  if (!pendingOrder) {
    console.log('⚠️ No pending orders found, using first available order');
    const testOrder = initialOrders[0];
    console.log(`🎯 Testing with order: ${testOrder.orderNumber} (current status: ${testOrder.orderStatus})`);
    
    // Test updating to a different status
    const newStatus = testOrder.orderStatus === 'delivered' ? 'pending' : 'processing';
    
    try {
      await testAPIUpdate(testOrder.orderNumber, newStatus);
      
      // Check database after update
      console.log('\n📊 Database State After Update:');
      const updatedOrders = checkDatabaseState();
      const updatedOrder = updatedOrders.find(o => o.orderNumber === testOrder.orderNumber);
      
      if (updatedOrder && updatedOrder.orderStatus === newStatus) {
        console.log('✅ DATABASE UPDATE SUCCESSFUL!');
        console.log(`   Order ${testOrder.orderNumber} status changed to: ${updatedOrder.orderStatus}`);
        console.log(`   Updated at: ${updatedOrder.updatedAt}`);
      } else {
        console.log('❌ DATABASE UPDATE FAILED!');
        console.log(`   Expected: ${newStatus}, Got: ${updatedOrder?.orderStatus || 'not found'}`);
      }
    } catch (error) {
      console.error('❌ Test failed:', error.message);
    }
  } else {
    console.log(`🎯 Testing with pending order: ${pendingOrder.orderNumber}`);
    
    try {
      await testAPIUpdate(pendingOrder.orderNumber, 'processing');
      
      // Check database after update
      console.log('\n📊 Database State After Update:');
      const updatedOrders = checkDatabaseState();
      const updatedOrder = updatedOrders.find(o => o.orderNumber === pendingOrder.orderNumber);
      
      if (updatedOrder && updatedOrder.orderStatus === 'processing') {
        console.log('✅ DATABASE UPDATE SUCCESSFUL!');
        console.log(`   Order ${pendingOrder.orderNumber} status changed to: ${updatedOrder.orderStatus}`);
        console.log(`   Updated at: ${updatedOrder.updatedAt}`);
      } else {
        console.log('❌ DATABASE UPDATE FAILED!');
        console.log(`   Expected: processing, Got: ${updatedOrder?.orderStatus || 'not found'}`);
      }
    } catch (error) {
      console.error('❌ Test failed:', error.message);
    }
  }
}

// Wait for server to be ready, then test
console.log('⏳ Waiting for server to be ready...');
setTimeout(testButtonFunctionality, 3000);