// Authentication utilities

export interface AuthUser {
  id: number;
  email: string;
  createdAt: Date;
}

export class AuthService {
  private static readonly USER_KEY = 'auth_user';

  // Store user data in localStorage (client-side only)
  static setUser(user: AuthUser): void {
    if (typeof window !== 'undefined') {
      localStorage.setItem(this.USER_KEY, JSON.stringify(user));
    }
  }

  // Get user data from localStorage
  static getUser(): AuthUser | null {
    if (typeof window !== 'undefined') {
      const userData = localStorage.getItem(this.USER_KEY);
      return userData ? JSON.parse(userData) : null;
    }
    return null;
  }

  // Remove user data (logout)
  static removeUser(): void {
    if (typeof window !== 'undefined') {
      localStorage.removeItem(this.USER_KEY);
    }
  }

  // Check if user is authenticated
  static isAuthenticated(): boolean {
    return this.getUser() !== null;
  }

  // Login user
  static async login(email: string, password: string): Promise<AuthUser> {
    const response = await fetch('/api/auth/login', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ email, password }),
    });

    const data = await response.json();

    if (!response.ok) {
      throw new Error(data.error || 'Login failed');
    }

    this.setUser(data.user);
    return data.user;
  }

  // Register user
  static async register(email: string, password: string): Promise<AuthUser> {
    const response = await fetch('/api/auth/register', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ email, password }),
    });

    const data = await response.json();

    if (!response.ok) {
      throw new Error(data.error || 'Registration failed');
    }

    this.setUser(data.user);
    return data.user;
  }

  // Logout user
  static logout(): void {
    this.removeUser();
  }
}