'use client';

import React, { createContext, useContext, useState, useEffect, ReactNode } from 'react';

export interface CartItem {
  id: number;
  name: string;
  slug: string;
  price: number;
  comparePrice?: number;
  quantity: number;
  image?: string;
  variantId?: string;
  variantSku?: string;
  variantAttributes?: { [key: string]: string };
}

interface CartContextType {
  items: CartItem[];
  itemCount: number;
  totalPrice: number;
  addItem: (item: Omit<CartItem, 'quantity'>, quantity?: number) => void;
  removeItem: (itemId: number, variantId?: string) => void;
  updateQuantity: (itemId: number, quantity: number, variantId?: string) => void;
  clearCart: () => void;
}

const CartContext = createContext<CartContextType | undefined>(undefined);

export const useCart = () => {
  const context = useContext(CartContext);
  if (context === undefined) {
    throw new Error('useCart must be used within a CartProvider');
  }
  return context;
};

interface CartProviderProps {
  children: ReactNode;
}

export const CartProvider: React.FC<CartProviderProps> = ({ children }) => {
  const [items, setItems] = useState<CartItem[]>([]);
  const [isLoaded, setIsLoaded] = useState(false);

  // Load cart from localStorage on mount
  useEffect(() => {
    try {
      const savedCart = localStorage.getItem('cart');
      if (savedCart) {
        setItems(JSON.parse(savedCart));
      }
    } catch (error) {
      console.error('Error loading cart from localStorage:', error);
    } finally {
      setIsLoaded(true);
    }
  }, []);

  // Save cart to localStorage whenever items change
  useEffect(() => {
    if (isLoaded) {
      localStorage.setItem('cart', JSON.stringify(items));
    }
  }, [items, isLoaded]);

  const addItem = (item: Omit<CartItem, 'quantity'>, quantity = 1) => {
    setItems(currentItems => {
      // For variant products, we need to check both product id and variant
      const existingItemIndex = currentItems.findIndex(cartItem => {
        if (item.variantId) {
          // For variable products, match both product id and variant id
          return cartItem.id === item.id && cartItem.variantId === item.variantId;
        } else {
          // For simple products, just match product id
          return cartItem.id === item.id;
        }
      });
      
      if (existingItemIndex >= 0) {
        // Item already exists, update quantity
        return currentItems.map((cartItem, index) =>
          index === existingItemIndex
            ? { ...cartItem, quantity: cartItem.quantity + quantity }
            : cartItem
        );
      } else {
        // New item, add to cart
        return [...currentItems, { ...item, quantity }];
      }
    });
  };

  const removeItem = (itemId: number, variantId?: string) => {
    setItems(currentItems => currentItems.filter(item => {
      // If variantId is provided, we need to match both product id and variant id
      if (variantId) {
        return !(item.id === itemId && item.variantId === variantId);
      }
      // Otherwise just match by product id and ensure it doesn't have a variantId
      return !(item.id === itemId && !item.variantId);
    }));
  };

  const updateQuantity = (itemId: number, quantity: number, variantId?: string) => {
    if (quantity <= 0) {
      removeItem(itemId, variantId);
      return;
    }

    setItems(currentItems =>
      currentItems.map(item => {
        if (variantId) {
          // For variable products, match both product id and variant id
          return (item.id === itemId && item.variantId === variantId) ? { ...item, quantity } : item;
        } else {
          // For simple products, just match product id and ensure it doesn't have a variantId
          return (item.id === itemId && !item.variantId) ? { ...item, quantity } : item;
        }
      })
    );
  };

  const clearCart = () => {
    setItems([]);
  };

  const itemCount = items.reduce((total, item) => total + item.quantity, 0);
  const totalPrice = items.reduce((total, item) => total + (item.price * item.quantity), 0);

  const contextValue: CartContextType = {
    items,
    itemCount,
    totalPrice,
    addItem,
    removeItem,
    updateQuantity,
    clearCart,
  };

  return (
    <CartContext.Provider value={contextValue}>
      {children}
    </CartContext.Provider>
  );
};