'use client';

import { useState } from 'react';
import { useRouter, usePathname } from 'next/navigation';
import Link from 'next/link';
import { AuthService } from '@/lib/auth';

interface SubItem {
  name: string;
  href: string;
  id: string;
  icon: string;
}

interface NavigationItem {
  name: string;
  icon: string;
  href: string;
  id: string;
  current: boolean;
  hasSubMenu?: boolean;
  isOpen?: boolean;
  subItems?: SubItem[];
}

interface SidebarProps {
  sidebarOpen: boolean;
  setSidebarOpen: (open: boolean) => void;
  activeMenuItem: string;
  setActiveMenuItem: (itemId: string) => void;
}

export default function Sidebar({ sidebarOpen, setSidebarOpen, activeMenuItem, setActiveMenuItem }: SidebarProps) {
  const [categoryMenuOpen, setCategoryMenuOpen] = useState(false);
  const [subcategoryMenuOpen, setSubcategoryMenuOpen] = useState(false);
  const [productMenuOpen, setProductMenuOpen] = useState(false);
  const [orderMenuOpen, setOrderMenuOpen] = useState(false);
  const [shippingMenuOpen, setShippingMenuOpen] = useState(false);
  const [customersMenuOpen, setCustomersMenuOpen] = useState(false);
  const router = useRouter();
  const pathname = usePathname();

  // Determine which menus should be open based on current path
  const isProductPage = pathname?.includes('/product');
  const isOrderPage = pathname?.includes('/orders');
  const isShippingPage = pathname?.includes('/shipping');
  const isCustomerPage = pathname?.includes('/customers');
  const isCategoryPage = pathname?.includes('/category');
  const isSubcategoryPage = pathname?.includes('/subcategory');

  // Set initial menu states based on current page
  useState(() => {
    setProductMenuOpen(isProductPage || false);
    setOrderMenuOpen(isOrderPage || false);
    setShippingMenuOpen(isShippingPage || false);
    setCustomersMenuOpen(isCustomerPage || false);
    setCategoryMenuOpen(isCategoryPage || false);
    setSubcategoryMenuOpen(isSubcategoryPage || false);
  });

  const handleLogout = () => {
    AuthService.logout();
    router.push('/');
  };

  const handleMenuClick = (itemId: string, hasSubMenu: boolean, href?: string) => {
    if (hasSubMenu) {
      if (itemId === 'category') {
        setCategoryMenuOpen(!categoryMenuOpen);
      } else if (itemId === 'subcategory') {
        setSubcategoryMenuOpen(!subcategoryMenuOpen);
      } else if (itemId === 'product') {
        setProductMenuOpen(!productMenuOpen);
      } else if (itemId === 'orders') {
        setOrderMenuOpen(!orderMenuOpen);
      } else if (itemId === 'shipping') {
        setShippingMenuOpen(!shippingMenuOpen);
      } else if (itemId === 'customers') {
        setCustomersMenuOpen(!customersMenuOpen);
      }
    } else {
      setActiveMenuItem(itemId);
      // Close any open submenus when clicking other items
      if (itemId !== 'category') setCategoryMenuOpen(false);
      if (itemId !== 'subcategory') setSubcategoryMenuOpen(false);
      if (itemId !== 'product') setProductMenuOpen(false);
      if (itemId !== 'orders') setOrderMenuOpen(false);
      if (itemId !== 'shipping') setShippingMenuOpen(false);
      if (itemId !== 'customers') setCustomersMenuOpen(false);
      
      if (href && href.startsWith('/')) {
        router.push(href);
      }
    }
  };

  const navigationItems: NavigationItem[] = [
    { 
      name: 'Dashboard', 
      icon: '📊', 
      href: '/dashboard', 
      id: 'dashboard',
      current: activeMenuItem === 'dashboard'
    },
    { 
      name: 'Category', 
      icon: '📁', 
      href: '#category', 
      id: 'category',
      current: activeMenuItem === 'category',
      hasSubMenu: true,
      isOpen: categoryMenuOpen,
      subItems: [
        { name: 'Add New Category', href: '/dashboard/category/add-new', id: 'add-category', icon: '➕' },
        { name: 'List of Categories', href: '/dashboard/category/list', id: 'list-categories', icon: '📃' }
      ]
    },
    { 
      name: 'Subcategory', 
      icon: '📂', 
      href: '#subcategory', 
      id: 'subcategory',
      current: activeMenuItem === 'subcategory',
      hasSubMenu: true,
      isOpen: subcategoryMenuOpen,
      subItems: [
        { name: 'Add New Subcategory', href: '/dashboard/subcategory/add-new', id: 'add-subcategory', icon: '➕' },
        { name: 'List of Subcategories', href: '/dashboard/subcategory/list', id: 'list-subcategories', icon: '📂' }
      ]
    },
    { 
      name: 'Product', 
      icon: '📦', 
      href: '#product', 
      id: 'product',
      current: activeMenuItem === 'product',
      hasSubMenu: true,
      isOpen: productMenuOpen,
      subItems: [
        { name: 'Add New Product', href: '/dashboard/product/add-new', id: 'add-product', icon: '🛍️' },
        { name: 'List of Products', href: '/dashboard/product/list', id: 'list-products', icon: '📃' }
      ]
    },
    { 
      name: 'Orders', 
      icon: '🛒', 
      href: '#orders', 
      id: 'orders',
      current: activeMenuItem === 'orders',
      hasSubMenu: true,
      isOpen: orderMenuOpen,
      subItems: [
        { name: 'All Orders', href: '/dashboard/orders/list', id: 'all-orders', icon: '📋' },
        { name: 'Pending Orders', href: '/dashboard/orders/pending', id: 'pending-orders', icon: '⏳' },
        { name: 'Completed Orders', href: '/dashboard/orders/completed', id: 'completed-orders', icon: '✅' },
        { name: 'Cancelled Orders', href: '/dashboard/orders/cancelled', id: 'cancelled-orders', icon: '❌' },
        { name: 'Order Analytics', href: '/dashboard/orders/analytics', id: 'order-analytics', icon: '📊' }
      ]
    },
    { 
      name: 'Shipping', 
      icon: '🚚', 
      href: '#shipping', 
      id: 'shipping',
      current: activeMenuItem === 'shipping',
      hasSubMenu: true,
      isOpen: shippingMenuOpen,
      subItems: [
        { name: 'Shipping Methods', href: '/dashboard/shipping/list', id: 'shipping-list', icon: '📋' },
        { name: 'Add New Method', href: '/dashboard/shipping/add-new', id: 'add-shipping', icon: '➕' }
      ]
    },
    { 
      name: 'Analytics', 
      icon: '📈', 
      href: '#analytics', 
      id: 'analytics',
      current: activeMenuItem === 'analytics'
    },
    { 
      name: 'Customers', 
      icon: '👥', 
      href: '#customers', 
      id: 'customers',
      current: activeMenuItem === 'customers',
      hasSubMenu: true,
      isOpen: customersMenuOpen,
      subItems: [
        { name: 'All Customers', href: '/dashboard/customers/list', id: 'all-customers', icon: '👥' },
        { name: 'Add New Customer', href: '/dashboard/customers/add-new', id: 'add-customer', icon: '➕' }
      ]
    },
    { 
      name: 'Settings', 
      icon: '⚙️', 
      href: '#settings', 
      id: 'settings',
      current: activeMenuItem === 'settings'
    }
  ];

  return (
    <>
      {/* Sidebar */}
      <div className={`${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} fixed inset-y-0 left-0 z-50 w-64 bg-gray-900 transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0`}>
        <div className="flex items-center justify-center h-16 bg-gray-800">
          <h1 className="text-white text-xl font-bold">📱 MyApp</h1>
        </div>
        
        <nav className="mt-8 px-4">
          <ul className="space-y-2">
            {navigationItems.map((item) => (
              <li key={item.name}>
                <div>
                  <button
                    onClick={() => handleMenuClick(item.id, item.hasSubMenu || false, item.href)}
                    className={`group flex items-center justify-between w-full px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
                      item.current || (item.hasSubMenu && item.isOpen)
                        ? 'bg-blue-600 text-white'
                        : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                    }`}
                  >
                    <div className="flex items-center">
                      <span className="text-lg mr-3">{item.icon}</span>
                      {item.name}
                    </div>
                    {item.hasSubMenu && (
                      <svg
                        className={`ml-auto h-5 w-5 transform transition-transform ${item.isOpen ? 'rotate-180' : ''}`}
                        fill="none"
                        viewBox="0 0 24 24"
                        stroke="currentColor"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    )}
                  </button>
                  
                  {item.hasSubMenu && item.isOpen && (
                    <ul className="mt-2 space-y-1 pl-4">
                      {item.subItems?.map((subItem) => (
                        <li key={subItem.name}>
                          <Link
                            href={subItem.href}
                            onClick={() => {
                              setActiveMenuItem(subItem.id);
                              // Close sidebar on mobile after navigation
                              if (window.innerWidth < 1024) {
                                setSidebarOpen(false);
                              }
                            }}
                            className={`flex items-center px-4 py-2 text-sm rounded-lg transition-colors ${
                              pathname === subItem.href
                                ? 'bg-blue-700 text-white'
                                : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                            }`}
                          >
                            <span className="text-base mr-3">{subItem.icon}</span>
                            {subItem.name}
                          </Link>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </nav>
        
        <div className="absolute bottom-0 w-full p-4">
          <button
            onClick={handleLogout}
            className="w-full bg-red-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-red-700 transition-colors"
          >
            Logout
          </button>
        </div>
      </div>

      {/* Overlay for mobile */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 bg-gray-600 bg-opacity-75 z-40 lg:hidden"
          onClick={() => setSidebarOpen(false)}
        ></div>
      )}
    </>
  );
}