'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useCart } from '../../context/CartContext';

interface Category {
  $loki: number;
  id: string;
  name: string;
  description: string;
  icon: string;
  slug: string;
  isActive: boolean;
}

interface Subcategory {
  id: string;
  name: string;
  slug: string;
  categoryId: string;
  icon: string;
  isActive: boolean;
  parentCategory?: {
    id: string;
    name: string;
    icon: string;
    slug: string;
  };
}

interface ProductAttribute {
  id: string;
  name: string; // e.g., "Size", "Color", "Storage"
  values: string[]; // e.g., ["S", "M", "L", "XL"] or ["Red", "Blue", "Green"]
}

interface ProductVariant {
  id: string;
  sku: string;
  price: number;
  comparePrice?: number;
  stockQuantity: number;
  attributes: { [key: string]: string }; // e.g., { "Size": "M", "Color": "Red" }
  imageUrl?: string;
  isActive: boolean;
}

interface BaseProduct {
  id: string;
  name: string;
  description: string;
  categoryId: string;
  subcategoryId?: string;
  slug: string;
  isActive: boolean;
  isFeatured: boolean;
  images: string[];
  createdAt?: string;
  type: 'simple' | 'variable';
}

interface SimpleProduct extends BaseProduct {
  type: 'simple';
  price: number;
  comparePrice?: number;
  stockQuantity: number;
  sku: string;
  imageUrl: string;
}

interface VariableProduct extends BaseProduct {
  type: 'variable';
  attributes: ProductAttribute[];
  variants: ProductVariant[];
  priceRange: {
    min: number;
    max: number;
  };
}

type Product = SimpleProduct | VariableProduct;

// Helper functions for product types
const getProductPrice = (product: Product): number => {
  if (!product.type || product.type === 'simple') {
    return (product as any).price || 0;
  }
  return product.priceRange?.min || 0;
};

const getProductMaxPrice = (product: Product): number => {
  if (!product.type || product.type === 'simple') {
    return (product as any).price || 0;
  }
  return product.priceRange?.max || 0;
};

const getProductComparePrice = (product: Product): number | undefined => {
  if (!product.type || product.type === 'simple') {
    return (product as any).comparePrice;
  }
  return undefined;
};

const getProductStockQuantity = (product: Product): number => {
  if (!product.type || product.type === 'simple') {
    return (product as any).stockQuantity || 0;
  }
  // For variable products, sum up all variant stock quantities
  return product.variants?.reduce((total, variant) => total + (variant.stockQuantity || 0), 0) || 0;
};

const getProductImageUrl = (product: Product): string => {
  if (!product.type || product.type === 'simple') {
    return (product as any).imageUrl || '/placeholder-product.jpg';
  }
  // For variable products, use the first image or first variant with image
  return product.images?.[0] || product.variants?.find(v => v.imageUrl)?.imageUrl || '/placeholder-product.jpg';
};

const getProductDisplayPrice = (product: Product): string => {
  if (!product.type || product.type === 'simple') {
    const price = (product as any).price || 0;
    return `$${price.toFixed(2)}`;
  }
  const priceRange = product.priceRange;
  if (!priceRange) {
    return '$0.00';
  }
  const { min, max } = priceRange;
  return min === max ? `$${min.toFixed(2)}` : `$${min.toFixed(2)} - $${max.toFixed(2)}`;
};

export default function ShopPage() {
  const [products, setProducts] = useState<Product[]>([]);
  const [categories, setCategories] = useState<Category[]>([]);
  const [subcategories, setSubcategories] = useState<Subcategory[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  const [selectedSubcategory, setSelectedSubcategory] = useState<string>('all');
  const [priceRange, setPriceRange] = useState<{ min: number; max: number }>({ min: 0, max: 1000 });
  const [sortBy, setSortBy] = useState<'name' | 'price' | 'newest'>('name');
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [itemsPerPage, setItemsPerPage] = useState<number>(12);
  const { addItem, itemCount } = useCart();

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      setLoading(true);
      
      // Fetch all data in parallel
      const [productsRes, categoriesRes, subcategoriesRes] = await Promise.all([
        fetch('/api/products'),
        fetch('/api/categories'),
        fetch('/api/subcategories')
      ]);

      const [productsData, categoriesData, subcategoriesData] = await Promise.all([
        productsRes.json(),
        categoriesRes.json(),
        subcategoriesRes.json()
      ]);

      if (productsRes.ok && productsData.products) {
        // Filter only active products
        const activeProducts = productsData.products.filter((product: Product) => product.isActive);
        setProducts(activeProducts);
      }

      if (categoriesRes.ok && categoriesData.categories) {
        // Filter only active categories
        const activeCategories = categoriesData.categories.filter((cat: Category) => cat.isActive);
        setCategories(activeCategories);
      }

      if (subcategoriesRes.ok && subcategoriesData.subcategories) {
        // Filter only active subcategories
        const activeSubcategories = subcategoriesData.subcategories.filter((sub: Subcategory) => sub.isActive);
        setSubcategories(activeSubcategories);
      }

    } catch (error) {
      console.error('Error fetching shop data:', error);
    } finally {
      setLoading(false);
    }
  };

  // Filter and sort products
  const filteredProducts = products
    .filter(product => {
      const matchesCategory = selectedCategory === 'all' || product.categoryId === selectedCategory;
      const matchesSubcategory = selectedSubcategory === 'all' || product.subcategoryId === selectedSubcategory;
      const productPrice = getProductPrice(product);
      const productMaxPrice = getProductMaxPrice(product);
      const matchesPrice = productPrice >= priceRange.min && productMaxPrice <= priceRange.max;
      const matchesSearch = searchTerm === '' || 
        product.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        product.description.toLowerCase().includes(searchTerm.toLowerCase());
      
      return matchesCategory && matchesSubcategory && matchesPrice && matchesSearch;
    })
    .sort((a, b) => {
      switch (sortBy) {
        case 'price':
          return getProductPrice(a) - getProductPrice(b);
        case 'newest':
          return new Date(b.createdAt || '').getTime() - new Date(a.createdAt || '').getTime();
        default:
          return a.name.localeCompare(b.name);
      }
    });

  // Pagination calculations
  const totalPages = Math.ceil(filteredProducts.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const endIndex = startIndex + itemsPerPage;
  const currentProducts = filteredProducts.slice(startIndex, endIndex);

  // Reset to first page when filters change
  useEffect(() => {
    setCurrentPage(1);
  }, [selectedCategory, selectedSubcategory, priceRange, searchTerm, sortBy]);

  // Pagination handlers
  const goToPage = (page: number) => {
    setCurrentPage(page);
    // Scroll to top of products section
    const productsSection = document.getElementById('products-section');
    if (productsSection) {
      productsSection.scrollIntoView({ behavior: 'smooth' });
    }
  };

  const goToPreviousPage = () => {
    if (currentPage > 1) {
      goToPage(currentPage - 1);
    }
  };

  const goToNextPage = () => {
    if (currentPage < totalPages) {
      goToPage(currentPage + 1);
    }
  };

  // Generate page numbers for pagination
  const getPageNumbers = () => {
    const pageNumbers = [];
    const maxVisiblePages = 5;
    
    if (totalPages <= maxVisiblePages) {
      for (let i = 1; i <= totalPages; i++) {
        pageNumbers.push(i);
      }
    } else {
      if (currentPage <= 3) {
        for (let i = 1; i <= 5; i++) {
          pageNumbers.push(i);
        }
      } else if (currentPage >= totalPages - 2) {
        for (let i = totalPages - 4; i <= totalPages; i++) {
          pageNumbers.push(i);
        }
      } else {
        for (let i = currentPage - 2; i <= currentPage + 2; i++) {
          pageNumbers.push(i);
        }
      }
    }
    
    return pageNumbers;
  };

  // Get subcategories for selected category
  const availableSubcategories = selectedCategory === 'all' 
    ? subcategories 
    : subcategories.filter(sub => sub.categoryId === selectedCategory);

  const handleAddToCart = (product: Product) => {
    addItem({
      id: parseInt(product.id) || Date.now(), // Convert string ID to number or use timestamp
      name: product.name,
      slug: product.slug,
      price: getProductPrice(product),
      comparePrice: getProductComparePrice(product),
      image: getProductImageUrl(product)
    });
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-white">
      {/* Header */}
      <header className="bg-white shadow-md sticky top-0 z-50">
        {/* Top bar */}
        <div className="bg-gray-900 text-white py-2">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="flex justify-between text-sm">
              <span>🚚 Free shipping on orders over $50</span>
              <div className="flex space-x-4">
                <Link href="/login" className="hover:text-gray-300">Account</Link>
                <span>📞 1-800-SHOP-NOW</span>
              </div>
            </div>
          </div>
        </div>

        {/* Main header */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center py-4">
            {/* Logo */}
            <div className="flex items-center">
              <Link href="/" className="flex items-center">
                <span className="text-2xl font-bold text-blue-600">🛒 ShopEasy</span>
              </Link>
            </div>

            {/* Search bar */}
            <div className="hidden md:flex flex-1 max-w-lg mx-8">
              <div className="relative w-full">
                <input
                  type="text"
                  placeholder="Search products..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center">
                  <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                  </svg>
                </div>
              </div>
            </div>

            {/* Right side */}
            <div className="flex items-center space-x-4">
              <button className="text-gray-600 hover:text-blue-600">
                <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                </svg>
              </button>
              
              <Link href="/cart" className="relative text-gray-600 hover:text-blue-600">
                <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 3h2l.4 2M7 13h10l4-8H5.4m-2.4 0L3 3m4 10v6a1 1 0 001 1h8a1 1 0 001-1v-6m-9 0h10" />
                </svg>
                {itemCount > 0 && (
                  <span className="absolute -top-2 -right-2 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">
                    {itemCount}
                  </span>
                )}
              </Link>
            </div>
          </div>

          {/* Navigation */}
          <nav className="hidden md:flex space-x-8 pb-4">
            <Link href="/shop" className="text-blue-600 font-medium">🛍️ Shop</Link>
            {categories.slice(0, 4).map((category) => (
              <Link 
                key={category.$loki || category.slug} 
                href={`/category/${category.slug}`} 
                className="text-gray-700 hover:text-blue-600"
              >
                {category.icon} {category.name}
              </Link>
            ))}
          </nav>
        </div>
      </header>

      <div className="bg-gray-50 min-h-screen">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Filters Sidebar */}
          <div className="lg:w-1/4">
            <div className="bg-white rounded-lg shadow-sm border p-6 space-y-6">
              <h2 className="text-lg font-semibold text-gray-900">Filters</h2>
              
              {/* Search */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Search</label>
                <input
                  type="text"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  placeholder="Search products..."
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              {/* Category Filter */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Category</label>
                <select
                  value={selectedCategory}
                  onChange={(e) => {
                    setSelectedCategory(e.target.value);
                    setSelectedSubcategory('all'); // Reset subcategory when category changes
                  }}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500"
                >
                  <option value="all">All Categories</option>
                  {categories.map(category => (
                    <option key={category.id} value={category.id}>
                      {category.icon} {category.name}
                    </option>
                  ))}
                </select>
              </div>

              {/* Subcategory Filter */}
              {availableSubcategories.length > 0 && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Subcategory</label>
                  <select
                    value={selectedSubcategory}
                    onChange={(e) => setSelectedSubcategory(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="all">All Subcategories</option>
                    {availableSubcategories.map(subcategory => (
                      <option key={subcategory.id} value={subcategory.id}>
                        {subcategory.icon} {subcategory.name}
                      </option>
                    ))}
                  </select>
                </div>
              )}

              {/* Price Range */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Price Range: ${priceRange.min} - ${priceRange.max}
                </label>
                <div className="space-y-2">
                  <input
                    type="range"
                    min="0"
                    max="1000"
                    value={priceRange.min}
                    onChange={(e) => setPriceRange({ ...priceRange, min: parseInt(e.target.value) })}
                    className="w-full"
                  />
                  <input
                    type="range"
                    min="0"
                    max="1000"
                    value={priceRange.max}
                    onChange={(e) => setPriceRange({ ...priceRange, max: parseInt(e.target.value) })}
                    className="w-full"
                  />
                </div>
              </div>

              {/* Sort */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Sort By</label>
                <select
                  value={sortBy}
                  onChange={(e) => setSortBy(e.target.value as any)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500"
                >
                  <option value="name">Name (A-Z)</option>
                  <option value="price">Price (Low to High)</option>
                  <option value="newest">Newest First</option>
                </select>
              </div>
            </div>
          </div>

          {/* Products Grid */}
          <div className="lg:w-3/4" id="products-section">
            {/* Results Header */}
            <div className="flex flex-col sm:flex-row sm:items-center justify-between mb-6 gap-4">
              <div>
                <h1 className="text-2xl font-bold text-gray-900">
                  Shop All Products
                </h1>
                <p className="text-gray-600">
                  {filteredProducts.length} products found
                  {filteredProducts.length > 0 && (
                    <span className="ml-2">
                      (Showing {startIndex + 1}-{Math.min(endIndex, filteredProducts.length)} of {filteredProducts.length})
                    </span>
                  )}
                </p>
              </div>
              
              {/* Items per page selector */}
              <div className="flex items-center space-x-2">
                <label className="text-sm text-gray-700">Show:</label>
                <select
                  value={itemsPerPage}
                  onChange={(e) => {
                    setItemsPerPage(parseInt(e.target.value));
                    setCurrentPage(1);
                  }}
                  className="px-3 py-1 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500"
                >
                  <option value={6}>6 per page</option>
                  <option value={12}>12 per page</option>
                  <option value={24}>24 per page</option>
                  <option value={48}>48 per page</option>
                </select>
              </div>
            </div>

            {/* Products Grid */}
            {filteredProducts.length === 0 ? (
              <div className="text-center py-12">
                <div className="text-6xl mb-4">🛍️</div>
                <h3 className="text-xl font-semibold text-gray-900 mb-2">No products found</h3>
                <p className="text-gray-600">Try adjusting your filters or search terms</p>
              </div>
            ) : (
              <>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                    {currentProducts.map((product) => (
                      <div key={product.id} className="bg-white rounded-2xl shadow-md hover:shadow-xl transition-shadow overflow-hidden group">
                        <Link href={`/product/${product.slug}`}>
                          <div className="relative cursor-pointer">
                            <div className="h-48 bg-gray-200 flex items-center justify-center">
                              <img
                                src={getProductImageUrl(product)}
                                alt={product.name}
                                className="h-48 w-full object-cover object-center group-hover:opacity-75"
                              />
                            </div>
                            <div className="absolute top-3 left-3">
                              <span className="bg-red-500 text-white px-2 py-1 text-xs font-semibold rounded">
                                {getProductComparePrice(product) && getProductComparePrice(product)! > getProductPrice(product) ? 'Sale' : 
                                 getProductStockQuantity(product) < 10 ? 'Limited' : 
                                 product.isFeatured ? 'Featured' : 'New'}
                              </span>
                            </div>
                          </div>
                        </Link>
                        <button className="absolute top-3 right-3 bg-white rounded-full p-2 shadow-md hover:bg-gray-50 z-10">
                          <svg className="h-4 w-4 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                          </svg>
                        </button>
                        
                        <div className="p-4">
                          <Link href={`/product/${product.slug}`}>
                            <h3 className="font-semibold text-gray-900 mb-2 group-hover:text-blue-600 transition-colors cursor-pointer">
                              {product.name}
                            </h3>
                          </Link>
                          
                          <div className="flex items-center mb-2">
                            <div className="flex text-yellow-400 text-sm">
                              {Array.from({length: 5}, (_, i) => (
                                <span key={i}>{i < Math.floor(4.0 + ((product.name.split('').reduce((a, b) => a + b.charCodeAt(0), 0) % 10) / 10)) ? '★' : '☆'}</span>
                              ))}
                            </div>
                            <span className="text-gray-500 text-sm ml-2">
                              ({50 + (product.name.split('').reduce((a, b) => a + b.charCodeAt(0), 0) % 200)})
                            </span>
                          </div>
                          
                          <div className="flex items-center justify-between mb-4">
                            <div>
                              <span className="text-2xl font-bold text-gray-900">{getProductDisplayPrice(product)}</span>
                              {getProductComparePrice(product) && getProductComparePrice(product)! > getProductPrice(product) && (
                                <span className="text-gray-500 line-through ml-2">${getProductComparePrice(product)!.toFixed(2)}</span>
                              )}
                            </div>
                          </div>
                          
                          <button 
                            onClick={() => handleAddToCart(product)}
                            className="w-full bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 transition-colors font-medium"
                          >
                            Add to Cart
                          </button>
                          
                          {getProductStockQuantity(product) <= 5 && getProductStockQuantity(product) > 0 && (
                            <p className="text-xs text-orange-600 mt-2">
                              Only {getProductStockQuantity(product)} left in stock
                            </p>
                          )}
                          {getProductStockQuantity(product) === 0 && (
                            <p className="text-xs text-red-600 mt-2">Out of stock</p>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>

                  {/* Pagination */}
                  {totalPages > 1 && (
                    <div className="mt-8 flex flex-col sm:flex-row items-center justify-between">
                      {/* Page info */}
                      <div className="text-sm text-gray-700 mb-4 sm:mb-0">
                        Showing {startIndex + 1} to {Math.min(endIndex, filteredProducts.length)} of {filteredProducts.length} results
                      </div>

                      {/* Pagination controls */}
                      <div className="flex items-center space-x-2">
                        {/* Previous button */}
                        <button
                          onClick={goToPreviousPage}
                          disabled={currentPage === 1}
                          className={`px-3 py-2 text-sm font-medium rounded-md ${
                            currentPage === 1
                              ? 'text-gray-400 cursor-not-allowed'
                              : 'text-gray-700 hover:text-blue-600 hover:bg-gray-50'
                          }`}
                        >
                          ← Previous
                        </button>

                        {/* Page numbers */}
                        <div className="flex items-center space-x-1">
                          {currentPage > 3 && totalPages > 5 && (
                            <>
                              <button
                                onClick={() => goToPage(1)}
                                className="px-3 py-2 text-sm font-medium text-gray-700 hover:text-blue-600 hover:bg-gray-50 rounded-md"
                              >
                                1
                              </button>
                              <span className="px-2 text-gray-500">...</span>
                            </>
                          )}

                          {getPageNumbers().map((page) => (
                            <button
                              key={page}
                              onClick={() => goToPage(page)}
                              className={`px-3 py-2 text-sm font-medium rounded-md ${
                                currentPage === page
                                  ? 'bg-blue-600 text-white'
                                  : 'text-gray-700 hover:text-blue-600 hover:bg-gray-50'
                              }`}
                            >
                              {page}
                            </button>
                          ))}

                          {currentPage < totalPages - 2 && totalPages > 5 && (
                            <>
                              <span className="px-2 text-gray-500">...</span>
                              <button
                                onClick={() => goToPage(totalPages)}
                                className="px-3 py-2 text-sm font-medium text-gray-700 hover:text-blue-600 hover:bg-gray-50 rounded-md"
                              >
                                {totalPages}
                              </button>
                            </>
                          )}
                        </div>

                        {/* Next button */}
                        <button
                          onClick={goToNextPage}
                          disabled={currentPage === totalPages}
                          className={`px-3 py-2 text-sm font-medium rounded-md ${
                            currentPage === totalPages
                              ? 'text-gray-400 cursor-not-allowed'
                              : 'text-gray-700 hover:text-blue-600 hover:bg-gray-50'
                          }`}
                        >
                          Next →
                        </button>
                      </div>
                    </div>
                  )}
                </>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-8">
            <div>
              <div className="flex items-center mb-4">
                <span className="text-2xl font-bold">🛒 ShopEasy</span>
              </div>
              <p className="text-gray-400 mb-4">Your one-stop shop for everything you need. Quality products, great prices, fast delivery.</p>
              <div className="flex space-x-4">
                <button className="text-gray-400 hover:text-white">📘</button>
                <button className="text-gray-400 hover:text-white">🐦</button>
                <button className="text-gray-400 hover:text-white">📷</button>
              </div>
            </div>
            
            <div>
              <h3 className="font-semibold mb-4">Customer Service</h3>
              <ul className="space-y-2 text-gray-400">
                <li><Link href="#" className="hover:text-white">Contact Us</Link></li>
                <li><Link href="#" className="hover:text-white">FAQ</Link></li>
                <li><Link href="#" className="hover:text-white">Shipping Info</Link></li>
                <li><Link href="#" className="hover:text-white">Returns</Link></li>
              </ul>
            </div>
            
            <div>
              <h3 className="font-semibold mb-4">Quick Links</h3>
              <ul className="space-y-2 text-gray-400">
                <li><Link href="/login" className="hover:text-white">My Account</Link></li>
                <li><Link href="#" className="hover:text-white">Order Tracking</Link></li>
                <li><Link href="#" className="hover:text-white">Wishlist</Link></li>
                <li><Link href="#" className="hover:text-white">Gift Cards</Link></li>
              </ul>
            </div>
            
            <div>
              <h3 className="font-semibold mb-4">Company</h3>
              <ul className="space-y-2 text-gray-400">
                <li><Link href="#" className="hover:text-white">About Us</Link></li>
                <li><Link href="#" className="hover:text-white">Careers</Link></li>
                <li><Link href="#" className="hover:text-white">Privacy Policy</Link></li>
                <li><Link href="#" className="hover:text-white">Terms of Service</Link></li>
              </ul>
            </div>
          </div>
          
          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2025 ShopEasy. All rights reserved. | 🚚 Free shipping on orders over $50</p>
          </div>
        </div>
      </footer>
    </div>
  );
}