'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useCart } from '../context/CartContext';

interface Category {
  $loki?: number;
  name: string;
  icon: string;
  slug: string;
  isActive: boolean;
}

// Enhanced product type interfaces
interface ProductAttribute {
  id: string;
  name: string;
  values: string[];
}

interface ProductVariant {
  id: string;
  attributes: { [key: string]: string };
  price: number;
  comparePrice?: number;
  stockQuantity: number;
  sku: string;
  isActive: boolean;
}

interface BaseProduct {
  $loki?: number;
  id: number;
  name: string;
  description: string;
  slug: string;
  categoryName: string;
  isActive: boolean;
  isFeatured: boolean;
  tags: string[];
  createdAt: string;
}

interface SimpleProduct extends BaseProduct {
  type: 'simple';
  price: number;
  comparePrice?: number;
  stockQuantity: number;
  sku: string;
}

interface VariableProduct extends BaseProduct {
  type: 'variable';
  attributes: ProductAttribute[];
  variants: ProductVariant[];
  priceRange?: {
    min: number;
    max: number;
  };
  compareAtRange?: {
    min: number;
    max: number;
  };
}

type Product = SimpleProduct | VariableProduct;

// Helper functions for product type handling
const getProductPrice = (product: Product): number => {
  if (product.type === 'simple') {
    return product.price;
  }
  // For variable products, return the minimum price from priceRange or variants
  if (product.priceRange?.min !== undefined) {
    return product.priceRange.min;
  }
  // Fallback to calculating from variants
  if (product.variants && product.variants.length > 0) {
    const minPrice = Math.min(...product.variants.map(v => v.price));
    return minPrice;
  }
  return 0; // Fallback
};

const getProductComparePrice = (product: Product): number | undefined => {
  if (product.type === 'simple') {
    return product.comparePrice;
  }
  // For variable products, return the minimum compare price
  if (product.compareAtRange?.min !== undefined) {
    return product.compareAtRange.min;
  }
  // Fallback to calculating from variants
  if (product.variants && product.variants.length > 0) {
    const comparePrices = product.variants
      .filter(v => v.comparePrice)
      .map(v => v.comparePrice!);
    if (comparePrices.length > 0) {
      return Math.min(...comparePrices);
    }
  }
  return undefined;
};

const getProductDisplayPrice = (product: Product): string => {
  if (product.type === 'simple') {
    return `$${product.price.toFixed(2)}`;
  }
  // For variable products, show price range if different min/max
  const minPrice = getProductPrice(product);
  const maxPrice = product.priceRange?.max !== undefined 
    ? product.priceRange.max 
    : (product.variants && product.variants.length > 0 
       ? Math.max(...product.variants.map(v => v.price)) 
       : minPrice);
  
  if (minPrice === maxPrice) {
    return `$${minPrice.toFixed(2)}`;
  }
  return `$${minPrice.toFixed(2)} - $${maxPrice.toFixed(2)}`;
};

const isProductOnSale = (product: Product): boolean => {
  const price = getProductPrice(product);
  const comparePrice = getProductComparePrice(product);
  return comparePrice !== undefined && comparePrice > price;
};

export default function Home() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const { itemCount, addItem } = useCart();
  const [categories, setCategories] = useState<Category[]>([]);
  const [featuredProducts, setFeaturedProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      // Fetch categories
      const categoriesResponse = await fetch('/api/categories');
      const categoriesData = await categoriesResponse.json();
      
      // Fetch products
      const productsResponse = await fetch('/api/products');
      const productsData = await productsResponse.json();
      
      if (categoriesResponse.ok && categoriesData.categories) {
        // Filter only active categories
        const activeCategories = categoriesData.categories.filter((cat: Category) => cat.isActive);
        setCategories(activeCategories);
      }
      
      if (productsResponse.ok && productsData.products) {
        // Filter only active and featured products
        const activeFeaturedProducts = productsData.products.filter(
          (product: Product) => product.isActive && product.isFeatured
        );
        setFeaturedProducts(activeFeaturedProducts);
      }
    } catch (error) {
      console.error('Error fetching data:', error);
    } finally {
      setLoading(false);
    }
  };

  const generateProductRating = (name: string) => {
    // Generate consistent rating based on product name for demo
    const hash = name.split('').reduce((a, b) => a + b.charCodeAt(0), 0);
    return 4.0 + ((hash % 10) / 10); // Rating between 4.0 and 4.9
  };

  const generateReviewCount = (name: string) => {
    // Generate consistent review count based on product name for demo
    const hash = name.split('').reduce((a, b) => a + b.charCodeAt(0), 0);
    return 50 + (hash % 200); // Review count between 50 and 250
  };

  const getProductBadge = (product: Product) => {
    if (isProductOnSale(product)) {
      return 'Sale';
    }
    const stockQuantity = product.type === 'simple' 
      ? product.stockQuantity 
      : (product.variants ? product.variants.reduce((sum, v) => sum + v.stockQuantity, 0) : 0);
    if (stockQuantity < 10) {
      return 'Limited';
    }
    if (product.isFeatured) {
      return 'Featured';
    }
    return 'New';
  };

  const testimonials = [
    {
      name: 'Sarah Johnson',
      avatar: '👩‍💼',
      review: 'Amazing quality and fast shipping! Will definitely order again.',
      rating: 5
    },
    {
      name: 'Mike Chen',
      avatar: '👨‍💻',
      review: 'Great customer service and excellent products.',
      rating: 5
    },
    {
      name: 'Emily Davis',
      avatar: '👩‍🎨',
      review: 'Love the variety and competitive prices.',
      rating: 4
    }
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Header */}
      <header className="bg-white shadow-md sticky top-0 z-50">
        {/* Top bar */}
        <div className="bg-gray-900 text-white py-2">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="flex justify-between text-sm">
              <span>🚚 Free shipping on orders over $50</span>
              <div className="flex space-x-4">
                <Link href="/login" className="hover:text-gray-300">Account</Link>
                <span>📞 1-800-SHOP-NOW</span>
              </div>
            </div>
          </div>
        </div>

        {/* Main header */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center py-4">
            {/* Logo */}
            <div className="flex items-center">
              <Link href="/" className="flex items-center">
                <span className="text-2xl font-bold text-blue-600">🛒 ShopEasy</span>
              </Link>
            </div>

            {/* Search bar */}
            <div className="hidden md:flex flex-1 max-w-lg mx-8">
              <div className="relative w-full">
                <input
                  type="text"
                  placeholder="Search products..."
                  className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center">
                  <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                  </svg>
                </div>
              </div>
            </div>

            {/* Right side */}
            <div className="flex items-center space-x-4">
              <button className="text-gray-600 hover:text-blue-600">
                <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                </svg>
              </button>
              
              <Link href="/cart" className="relative text-gray-600 hover:text-blue-600">
                <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 3h2l.4 2M7 13h10l4-8H5.4m-2.4 0L3 3m4 10v6a1 1 0 001 1h8a1 1 0 001-1v-6m-9 0h10" />
                </svg>
                {itemCount > 0 && (
                  <span className="absolute -top-2 -right-2 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">
                    {itemCount}
                  </span>
                )}
              </Link>

              {/* Mobile menu button */}
              <button
                className="md:hidden text-gray-600"
                onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              >
                <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
            </div>
          </div>

          {/* Navigation */}
          <nav className="hidden md:flex space-x-8 pb-4">
            <Link href="/shop" className="text-gray-700 hover:text-blue-600 font-medium">🛍️ Shop</Link>
            <Link href="#categories" className="text-gray-700 hover:text-blue-600 font-medium">All Categories</Link>
            {categories.slice(0, 3).map((category) => (
              <Link 
                key={category.$loki || category.slug} 
                href={`/category/${category.slug}`} 
                className="text-gray-700 hover:text-blue-600"
              >
                {category.icon} {category.name}
              </Link>
            ))}
            <Link href="#featured" className="text-gray-700 hover:text-blue-600">Featured</Link>
          </nav>
        </div>

        {/* Mobile menu */}
        {mobileMenuOpen && (
          <div className="md:hidden bg-white border-t border-gray-200">
            <div className="px-4 pt-2 pb-3 space-y-1">
              <input
                type="text"
                placeholder="Search products..."
                className="w-full px-3 py-2 border border-gray-300 rounded-md mb-3"
              />
              <Link href="/shop" className="block px-3 py-2 text-gray-700 font-medium">🛍️ Shop</Link>
              <Link href="#categories" className="block px-3 py-2 text-gray-700">All Categories</Link>
              {categories.slice(0, 3).map((category) => (
                <Link 
                  key={category.$loki || category.slug} 
                  href={`/category/${category.slug}`} 
                  className="block px-3 py-2 text-gray-700"
                >
                  {category.icon} {category.name}
                </Link>
              ))}
              <Link href="#featured" className="block px-3 py-2 text-gray-700">Featured</Link>
            </div>
          </div>
        )}
      </header>

      {/* Hero Section */}
      <section className="relative bg-gradient-to-r from-blue-600 to-purple-700 text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 items-center">
            <div>
              <h1 className="text-4xl md:text-6xl font-bold mb-4">
                Shop Smart,
                <span className="block text-yellow-300">Save More</span>
              </h1>
              <p className="text-xl mb-8 opacity-90">
                Discover amazing deals on top-quality products. Free shipping on orders over $50!
              </p>
              <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
                <Link href="/shop" className="bg-yellow-400 text-gray-900 px-8 py-3 rounded-lg font-semibold hover:bg-yellow-300 transition-colors text-center">
                  Shop Now
                </Link>
                <Link href="/shop" className="border-2 border-white text-white px-8 py-3 rounded-lg font-semibold hover:bg-white hover:text-blue-600 transition-colors text-center">
                  View Deals
                </Link>
              </div>
            </div>
            <div className="relative">
              <div className="bg-white bg-opacity-10 rounded-2xl p-8 backdrop-blur">
                <div className="text-center">
                  <div className="text-6xl mb-4">🛍️</div>
                  <h3 className="text-2xl font-bold mb-2">50% OFF</h3>
                  <p className="text-lg">On selected items</p>
                  <div className="mt-4 text-sm opacity-75">Limited time offer</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Categories Section */}
      <section className="py-16 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Shop by Category</h2>
            <p className="text-lg text-gray-600">Find exactly what you're looking for</p>
          </div>
          
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            {categories.map((category, index) => (
              <Link key={index} href={`/category/${category.slug}`} className="group cursor-pointer">
                <div className="bg-white rounded-2xl shadow-md overflow-hidden hover:shadow-xl transition-shadow">
                  <div className="h-32 bg-gradient-to-br from-blue-400 to-purple-600 flex items-center justify-center">
                    <span className="text-4xl">{category.icon}</span>
                  </div>
                  <div className="p-4 text-center">
                    <h3 className="font-semibold text-gray-900 group-hover:text-blue-600 transition-colors">
                      {category.name}
                    </h3>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </div>
      </section>

      {/* Featured Products */}
      <section className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Featured Products</h2>
            <p className="text-lg text-gray-600">Our top picks just for you</p>
          </div>
          
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
            {featuredProducts.map((product) => (
              <div key={product.id} className="bg-white rounded-2xl shadow-md hover:shadow-xl transition-shadow overflow-hidden group">
                <Link href={`/product/${product.slug}`}>
                  <div className="relative cursor-pointer">
                    <div className="h-48 bg-gray-200 flex items-center justify-center">
                      <span className="text-4xl">📦</span>
                    </div>
                    <div className="absolute top-3 left-3">
                      <span className="bg-red-500 text-white px-2 py-1 text-xs font-semibold rounded">
                        {getProductBadge(product)}
                      </span>
                    </div>
                  </div>
                </Link>
                <button className="absolute top-3 right-3 bg-white rounded-full p-2 shadow-md hover:bg-gray-50 z-10">
                  <svg className="h-4 w-4 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                  </svg>
                </button>
                
                <div className="p-4">
                  <Link href={`/product/${product.slug}`}>
                    <h3 className="font-semibold text-gray-900 mb-2 group-hover:text-blue-600 transition-colors cursor-pointer">
                      {product.name}
                    </h3>
                  </Link>
                  
                  <div className="flex items-center mb-2">
                    <div className="flex text-yellow-400 text-sm">
                      {'★'.repeat(Math.floor(generateProductRating(product.name)))}
                      {'☆'.repeat(5 - Math.floor(generateProductRating(product.name)))}
                    </div>
                    <span className="text-gray-500 text-sm ml-2">({generateReviewCount(product.name)})</span>
                  </div>
                  
                  <div className="flex items-center justify-between mb-4">
                    <div>
                      <span className="text-2xl font-bold text-gray-900">{getProductDisplayPrice(product)}</span>
                      {isProductOnSale(product) && (
                        <span className="text-gray-500 line-through ml-2">${getProductComparePrice(product)?.toFixed(2)}</span>
                      )}
                    </div>
                  </div>
                  
<div className="flex gap-2">
                    <button 
                      onClick={() => {
                        if (product.type === 'variable' && product.variants && product.variants.length > 0) {
                          // For variable products, add the first available variant to cart
                          const firstVariant = product.variants.find(v => v.isActive && v.stockQuantity > 0) || product.variants[0];
                          const variantAttributesText = Object.entries(firstVariant.attributes)
                            .map(([key, value]) => `${key}: ${value}`)
                            .join(', ');
                          
                          addItem({
                            id: product.id,
                            name: `${product.name} (${variantAttributesText})`,
                            slug: product.slug,
                            price: firstVariant.price,
                            comparePrice: firstVariant.comparePrice,
                            variantId: firstVariant.id,
                            variantSku: firstVariant.sku,
                            variantAttributes: firstVariant.attributes
                          });
                        } else {
                          // For simple products, add normally
                          addItem({
                            id: product.id,
                            name: product.name,
                            slug: product.slug,
                            price: getProductPrice(product),
                            comparePrice: getProductComparePrice(product)
                          });
                        }
                      }}
                      className={`${product.type === 'variable' ? 'flex-1' : 'w-full'} bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 transition-colors font-medium`}
                    >
                      Add to Cart
                    </button>
                    {product.type === 'variable' && (
                      <Link 
                        href={`/product/${product.slug}`}
                        className="bg-gray-100 text-gray-800 px-3 py-2 rounded-lg hover:bg-gray-200 transition-colors text-center flex items-center justify-center"
                        title="View all options"
                      >
                        ⚙️
                      </Link>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
          
          <div className="text-center mt-12">
            <Link href="/shop" className="bg-gray-100 text-gray-900 px-8 py-3 rounded-lg hover:bg-gray-200 transition-colors font-medium inline-block">
              View All Products
            </Link>
          </div>
        </div>
      </section>

      {/* Testimonials */}
      <section className="py-16 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">What Our Customers Say</h2>
            <p className="text-lg text-gray-600">Join thousands of satisfied customers</p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {testimonials.map((testimonial, index) => (
              <div key={index} className="bg-white rounded-2xl p-6 shadow-md">
                <div className="flex items-center mb-4">
                  <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center text-2xl">
                    {testimonial.avatar}
                  </div>
                  <div className="ml-4">
                    <h4 className="font-semibold text-gray-900">{testimonial.name}</h4>
                    <div className="flex text-yellow-400 text-sm">
                      {'★'.repeat(testimonial.rating)}
                      {'☆'.repeat(5 - testimonial.rating)}
                    </div>
                  </div>
                </div>
                <p className="text-gray-600 italic">"{testimonial.review}"</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Newsletter */}
      <section className="py-16 bg-blue-600">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl font-bold text-white mb-4">Stay Updated</h2>
          <p className="text-xl text-blue-100 mb-8">Subscribe to get special offers and updates</p>
          
          <div className="max-w-md mx-auto flex">
            <input
              type="email"
              placeholder="Enter your email"
              className="flex-1 px-4 py-3 rounded-l-lg focus:outline-none focus:ring-2 focus:ring-blue-300"
            />
            <button className="bg-yellow-400 text-gray-900 px-6 py-3 rounded-r-lg hover:bg-yellow-300 transition-colors font-medium">
              Subscribe
            </button>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-8">
            <div>
              <div className="flex items-center mb-4">
                <span className="text-2xl font-bold">🛒 ShopEasy</span>
              </div>
              <p className="text-gray-400 mb-4">Your one-stop shop for everything you need. Quality products, great prices, fast delivery.</p>
              <div className="flex space-x-4">
                <button className="text-gray-400 hover:text-white">📘</button>
                <button className="text-gray-400 hover:text-white">🐦</button>
                <button className="text-gray-400 hover:text-white">📷</button>
              </div>
            </div>
            
            <div>
              <h3 className="font-semibold mb-4">Customer Service</h3>
              <ul className="space-y-2 text-gray-400">
                <li><Link href="#" className="hover:text-white">Contact Us</Link></li>
                <li><Link href="#" className="hover:text-white">FAQ</Link></li>
                <li><Link href="#" className="hover:text-white">Shipping Info</Link></li>
                <li><Link href="#" className="hover:text-white">Returns</Link></li>
              </ul>
            </div>
            
            <div>
              <h3 className="font-semibold mb-4">Quick Links</h3>
              <ul className="space-y-2 text-gray-400">
                <li><Link href="/login" className="hover:text-white">My Account</Link></li>
                <li><Link href="#" className="hover:text-white">Order Tracking</Link></li>
                <li><Link href="#" className="hover:text-white">Wishlist</Link></li>
                <li><Link href="#" className="hover:text-white">Gift Cards</Link></li>
              </ul>
            </div>
            
            <div>
              <h3 className="font-semibold mb-4">Company</h3>
              <ul className="space-y-2 text-gray-400">
                <li><Link href="#" className="hover:text-white">About Us</Link></li>
                <li><Link href="#" className="hover:text-white">Careers</Link></li>
                <li><Link href="#" className="hover:text-white">Privacy Policy</Link></li>
                <li><Link href="#" className="hover:text-white">Terms of Service</Link></li>
              </ul>
            </div>
          </div>
          
          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2025 ShopEasy. All rights reserved. | 🚚 Free shipping on orders over $50</p>
          </div>
        </div>
      </footer>
    </div>
  );
}
