'use client';

import { useEffect, useState } from 'react';
import { AuthService, AuthUser } from '@/lib/auth';
import { useRouter } from 'next/navigation';
import Link from 'next/link';

interface ShippingMethod {
  id: number;
  name: string;
  description: string;
  cost: number;
  estimatedDays: string;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

export default function ShippingMethodsList() {
  const [user, setUser] = useState<AuthUser | null>(null);
  const [loading, setLoading] = useState(true);
  const [shippingMethods, setShippingMethods] = useState<ShippingMethod[]>([]);
  const [filteredMethods, setFilteredMethods] = useState<ShippingMethod[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [deleting, setDeleting] = useState<number | null>(null);
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [categoryMenuOpen, setCategoryMenuOpen] = useState(false);
  const [productMenuOpen, setProductMenuOpen] = useState(false);
  const [orderMenuOpen, setOrderMenuOpen] = useState(false);
  const [shippingMenuOpen, setShippingMenuOpen] = useState(true);
  const [customersMenuOpen, setCustomersMenuOpen] = useState(false);
  const [activeMenuItem, setActiveMenuItem] = useState('shipping-list');
  const router = useRouter();

  useEffect(() => {
    const currentUser = AuthService.getUser();
    if (!currentUser) {
      router.push('/login');
      return;
    }
    setUser(currentUser);
    fetchShippingMethods();
  }, [router]);

  const fetchShippingMethods = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/shipping');
      if (response.ok) {
        const methodsData = await response.json();
        setShippingMethods(methodsData);
        setFilteredMethods(methodsData);
      } else {
        console.error('Failed to fetch shipping methods');
        setShippingMethods([]);
        setFilteredMethods([]);
      }
    } catch (error) {
      console.error('Error fetching shipping methods:', error);
      setShippingMethods([]);
      setFilteredMethods([]);
    } finally {
      setLoading(false);
    }
  };

  const deleteShippingMethod = async (id: number, name: string) => {
    if (!confirm(`Are you sure you want to delete "${name}"? This action cannot be undone.`)) {
      return;
    }

    try {
      setDeleting(id);
      
      const response = await fetch(`/api/shipping/${id}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        // Remove from local state
        setShippingMethods(prev => prev.filter(method => method.id !== id));
        applyFilters(shippingMethods.filter(method => method.id !== id));
        alert(`Shipping method "${name}" deleted successfully`);
      } else {
        const error = await response.json();
        throw new Error(error.error || 'Failed to delete shipping method');
      }
    } catch (error) {
      console.error('Error deleting shipping method:', error);
      alert(`Failed to delete shipping method: ${error instanceof Error ? error.message : 'Unknown error'}`);
    } finally {
      setDeleting(null);
    }
  };

  const toggleMethodStatus = async (id: number, currentStatus: boolean) => {
    try {
      const method = shippingMethods.find(m => m.id === id);
      if (!method) return;

      const response = await fetch(`/api/shipping/${id}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          ...method,
          isActive: !currentStatus
        }),
      });

      if (response.ok) {
        // Update local state
        setShippingMethods(prev => prev.map(method => 
          method.id === id 
            ? { ...method, isActive: !currentStatus, updatedAt: new Date().toISOString() }
            : method
        ));
        
        // Re-apply filters
        const updatedMethods = shippingMethods.map(method => 
          method.id === id 
            ? { ...method, isActive: !currentStatus, updatedAt: new Date().toISOString() }
            : method
        );
        applyFilters(updatedMethods);
      } else {
        const error = await response.json();
        throw new Error(error.error || 'Failed to update shipping method');
      }
    } catch (error) {
      console.error('Error updating shipping method status:', error);
      alert(`Failed to update status: ${error instanceof Error ? error.message : 'Unknown error'}`);
    }
  };

  const applyFilters = (methodsToFilter: ShippingMethod[]) => {
    let filtered = methodsToFilter;

    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(method =>
        method.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        method.description.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    // Filter by status
    if (statusFilter !== 'all') {
      filtered = filtered.filter(method => 
        statusFilter === 'active' ? method.isActive : !method.isActive
      );
    }

    setFilteredMethods(filtered);
  };

  useEffect(() => {
    applyFilters(shippingMethods);
  }, [shippingMethods, searchTerm, statusFilter]);

  const handleLogout = () => {
    AuthService.logout();
    router.push('/');
  };

  const getStatusColor = (isActive: boolean) => {
    return isActive 
      ? 'bg-green-100 text-green-800' 
      : 'bg-red-100 text-red-800';
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-900">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  const navigationItems = [
    { 
      name: 'Dashboard', 
      icon: '📊', 
      href: '/dashboard', 
      id: 'dashboard',
      current: activeMenuItem === 'dashboard'
    },
    { 
      name: 'Category', 
      icon: '📁', 
      href: '#category', 
      id: 'category',
      current: activeMenuItem === 'category',
      hasSubMenu: true,
      isOpen: categoryMenuOpen,
      subItems: [
        { name: 'Add New Category', href: '/dashboard/category/add-new', id: 'add-category', icon: '➕' },
        { name: 'List of Categories', href: '/dashboard/category/list', id: 'list-categories', icon: '📃' }
      ]
    },
    { 
      name: 'Product', 
      icon: '📦', 
      href: '#product', 
      id: 'product',
      current: activeMenuItem === 'product',
      hasSubMenu: true,
      isOpen: productMenuOpen,
      subItems: [
        { name: 'Add New Product', href: '/dashboard/product/add-new', id: 'add-product', icon: '🛍️' },
        { name: 'List of Products', href: '/dashboard/product/list', id: 'list-products', icon: '📃' }
      ]
    },
    { 
      name: 'Orders', 
      icon: '🛒', 
      href: '#orders', 
      id: 'orders',
      current: activeMenuItem === 'orders',
      hasSubMenu: true,
      isOpen: orderMenuOpen,
      subItems: [
        { name: 'All Orders', href: '/dashboard/orders/list', id: 'all-orders', icon: '📋' },
        { name: 'Pending Orders', href: '/dashboard/orders/pending', id: 'pending-orders', icon: '⏳' },
        { name: 'Completed Orders', href: '/dashboard/orders/completed', id: 'completed-orders', icon: '✅' },
        { name: 'Cancelled Orders', href: '/dashboard/orders/cancelled', id: 'cancelled-orders', icon: '❌' },
        { name: 'Order Analytics', href: '/dashboard/orders/analytics', id: 'order-analytics', icon: '📊' }
      ]
    },
    { 
      name: 'Shipping', 
      icon: '🚚', 
      href: '#shipping', 
      id: 'shipping',
      current: activeMenuItem === 'shipping',
      hasSubMenu: true,
      isOpen: shippingMenuOpen,
      subItems: [
        { name: 'Shipping Methods', href: '/dashboard/shipping/list', id: 'shipping-list', icon: '📋' },
        { name: 'Add New Method', href: '/dashboard/shipping/add-new', id: 'add-shipping', icon: '➕' }
      ]
    },
    { 
      name: 'Analytics', 
      icon: '📈', 
      href: '#analytics', 
      id: 'analytics',
      current: activeMenuItem === 'analytics'
    },
    { 
      name: 'Customers', 
      icon: '👥', 
      href: '#customers', 
      id: 'customers',
      current: activeMenuItem === 'customers',
      hasSubMenu: true,
      isOpen: customersMenuOpen,
      subItems: [
        { name: 'All Customers', href: '/dashboard/customers/list', id: 'all-customers', icon: '👥' },
        { name: 'Add New Customer', href: '/dashboard/customers/add-new', id: 'add-customer', icon: '➕' }
      ]
    },
    { 
      name: 'Settings', 
      icon: '⚙️', 
      href: '#settings', 
      id: 'settings',
      current: activeMenuItem === 'settings'
    },
  ];

  const handleMenuClick = (itemId: string, hasSubMenu?: boolean) => {
    if (hasSubMenu) {
      if (itemId === 'category') {
        setCategoryMenuOpen(!categoryMenuOpen);
      } else if (itemId === 'product') {
        setProductMenuOpen(!productMenuOpen);
      } else if (itemId === 'orders') {
        setOrderMenuOpen(!orderMenuOpen);
      } else if (itemId === 'shipping') {
        setShippingMenuOpen(!shippingMenuOpen);
      } else if (itemId === 'customers') {
        setCustomersMenuOpen(!customersMenuOpen);
      }
    } else {
      setActiveMenuItem(itemId);
      // Close any open submenus when clicking other items
      if (itemId !== 'category') setCategoryMenuOpen(false);
      if (itemId !== 'product') setProductMenuOpen(false);
      if (itemId !== 'orders') setOrderMenuOpen(false);
      if (itemId !== 'shipping') setShippingMenuOpen(false);
      if (itemId !== 'customers') setCustomersMenuOpen(false);
    }
  };

  const handleSubMenuClick = (subItemId: string, parentId: string, href?: string) => {
    setActiveMenuItem(subItemId);
    // Keep the parent menu open
    if (parentId === 'category') {
      setCategoryMenuOpen(true);
    } else if (parentId === 'product') {
      setProductMenuOpen(true);
    } else if (parentId === 'orders') {
      setOrderMenuOpen(true);
    } else if (parentId === 'shipping') {
      setShippingMenuOpen(true);
    }
    
    // Navigate to actual route if href is provided
    if (href && href.startsWith('/')) {
      router.push(href);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 flex">
      {/* Sidebar */}
      <div className={`${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} fixed inset-y-0 left-0 z-50 w-64 bg-gray-900 transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0`}>
        <div className="flex items-center justify-center h-16 bg-gray-800">
          <h1 className="text-white text-xl font-bold">📱 MyApp</h1>
        </div>
        
        <nav className="mt-8 px-4">
          <ul className="space-y-2">
            {navigationItems.map((item) => (
              <li key={item.name}>
                <div>
                  <button
                    onClick={() => handleMenuClick(item.id, item.hasSubMenu)}
                    className={`group flex items-center justify-between w-full px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
                      item.current || (item.hasSubMenu && item.isOpen)
                        ? 'bg-blue-600 text-white'
                        : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                    }`}
                  >
                    <div className="flex items-center">
                      <span className="mr-3 text-lg">{item.icon}</span>
                      {item.name}
                    </div>
                    {item.hasSubMenu && (
                      <svg 
                        className={`w-4 h-4 transition-transform ${item.isOpen ? 'rotate-180' : ''}`}
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    )}
                  </button>
                  
                  {/* Submenu */}
                  {item.hasSubMenu && item.isOpen && item.subItems && (
                    <ul className="mt-2 space-y-1 pl-4">
                      {item.subItems.map((subItem) => (
                        <li key={subItem.id}>
                          <button
                            onClick={() => handleSubMenuClick(subItem.id, item.id, subItem.href)}
                            className={`group flex items-center w-full px-4 py-2 text-sm rounded-lg transition-colors ${
                              activeMenuItem === subItem.id
                                ? 'bg-blue-500 text-white'
                                : 'text-gray-400 hover:bg-gray-700 hover:text-white'
                            }`}
                          >
                            <span className="mr-3 text-base">{subItem.icon}</span>
                            {subItem.name}
                          </button>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </nav>
        
        <div className="absolute bottom-0 w-full p-4">
          <div className="bg-gray-800 rounded-lg p-4">
            <div className="flex items-center">
              <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center">
                <span className="text-white font-semibold text-sm">
                  {user?.email?.charAt(0).toUpperCase() || 'U'}
                </span>
              </div>
              <div className="ml-3">
                <p className="text-sm font-medium text-white truncate">{user?.email || 'User'}</p>
                <p className="text-xs text-gray-400">Online</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Sidebar overlay for mobile */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 z-40 bg-gray-600 bg-opacity-75 lg:hidden" 
          onClick={() => setSidebarOpen(false)}
        ></div>
      )}

      {/* Main content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Header */}
        <header className="bg-white shadow-sm border-b border-gray-200">
          <div className="flex items-center justify-between px-6 py-4">
            <div className="flex items-center">
              <button
                className="text-gray-500 hover:text-gray-600 lg:hidden"
                onClick={() => setSidebarOpen(true)}
              >
                <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
              <h1 className="ml-4 lg:ml-0 text-2xl font-semibold text-gray-900">Shipping Methods</h1>
            </div>
            
            <div className="flex items-center space-x-4">
              <Link
                href="/dashboard/shipping/add-new"
                className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors flex items-center gap-2"
              >
                <span>➕</span>
                Add New Method
              </Link>
              <button
                onClick={handleLogout}
                className="bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Logout
              </button>
            </div>
          </div>
        </header>

        {/* Main content area */}
        <main className="flex-1 overflow-y-auto bg-gray-50 p-6">
          {/* Filters and Search */}
          <div className="mb-6 bg-white rounded-lg shadow-sm p-6">
            <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
              <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
                <div className="relative">
                  <input
                    type="text"
                    placeholder="Search shipping methods..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full sm:w-64 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center">
                    <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                    </svg>
                  </div>
                </div>
                
                <select
                  value={statusFilter}
                  onChange={(e) => setStatusFilter(e.target.value)}
                  className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="all">All Status</option>
                  <option value="active">Active</option>
                  <option value="inactive">Inactive</option>
                </select>
              </div>
              
              <div className="text-sm text-gray-600">
                Showing {filteredMethods.length} of {shippingMethods.length} methods
              </div>
            </div>
          </div>

          {/* Shipping Methods Table */}
          <div className="bg-white rounded-lg shadow-sm overflow-hidden">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Method
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Cost
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Delivery Time
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Created
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {filteredMethods.map((method) => (
                    <tr key={method.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">
                            {method.name}
                          </div>
                          <div className="text-sm text-gray-500">
                            {method.description}
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">
                          {method.cost === 0 ? 'Free' : `$${method.cost.toFixed(2)}`}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {method.estimatedDays} {method.estimatedDays === '1' ? 'day' : 'days'}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <button
                          onClick={() => toggleMethodStatus(method.id, method.isActive)}
                          className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(method.isActive)} hover:opacity-80 transition-opacity`}
                        >
                          {method.isActive ? 'Active' : 'Inactive'}
                        </button>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {new Date(method.createdAt).toLocaleDateString()}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div className="flex items-center space-x-3">
                          <Link
                            href={`/dashboard/shipping/edit/${method.id}`}
                            className="text-blue-600 hover:text-blue-900"
                          >
                            Edit
                          </Link>
                          <button
                            onClick={() => deleteShippingMethod(method.id, method.name)}
                            disabled={deleting === method.id}
                            className="text-red-600 hover:text-red-900 disabled:opacity-50"
                          >
                            {deleting === method.id ? 'Deleting...' : 'Delete'}
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            
            {filteredMethods.length === 0 && (
              <div className="text-center py-12">
                <div className="text-4xl mb-4">🚚</div>
                <h3 className="text-lg font-medium text-gray-900 mb-2">No shipping methods found</h3>
                <p className="text-gray-500 mb-4">
                  {searchTerm || statusFilter !== 'all' 
                    ? 'Try adjusting your search criteria'
                    : 'Get started by creating your first shipping method'
                  }
                </p>
                <Link
                  href="/dashboard/shipping/add-new"
                  className="inline-flex items-center gap-2 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
                >
                  <span>➕</span>
                  Add New Method
                </Link>
              </div>
            )}
          </div>
        </main>
      </div>
    </div>
  );
}