'use client';

import { useState } from 'react';
import { AuthService, AuthUser } from '@/lib/auth';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import { useEffect } from 'react';

export default function AddNewShippingMethod() {
  const [user, setUser] = useState<AuthUser | null>(null);
  const [loading, setLoading] = useState(false);
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [categoryMenuOpen, setCategoryMenuOpen] = useState(false);
  const [productMenuOpen, setProductMenuOpen] = useState(false);
  const [orderMenuOpen, setOrderMenuOpen] = useState(false);
  const [shippingMenuOpen, setShippingMenuOpen] = useState(true);
  const [activeMenuItem, setActiveMenuItem] = useState('add-shipping');
  const router = useRouter();

  // Form state
  const [formData, setFormData] = useState({
    name: '',
    description: '',
    cost: '',
    estimatedDays: '',
    isActive: true
  });

  const [errors, setErrors] = useState<{[key: string]: string}>({});

  useEffect(() => {
    const currentUser = AuthService.getUser();
    if (!currentUser) {
      router.push('/login');
      return;
    }
    setUser(currentUser);
  }, [router]);

  const validateForm = () => {
    const newErrors: {[key: string]: string} = {};

    if (!formData.name.trim()) {
      newErrors.name = 'Method name is required';
    }

    if (!formData.description.trim()) {
      newErrors.description = 'Description is required';
    }

    if (!formData.cost.trim()) {
      newErrors.cost = 'Cost is required';
    } else {
      const cost = parseFloat(formData.cost);
      if (isNaN(cost) || cost < 0) {
        newErrors.cost = 'Cost must be a valid number (0 or greater)';
      }
    }

    if (!formData.estimatedDays.trim()) {
      newErrors.estimatedDays = 'Estimated delivery time is required';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) {
      return;
    }

    try {
      setLoading(true);

      const response = await fetch('/api/shipping', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          name: formData.name.trim(),
          description: formData.description.trim(),
          cost: parseFloat(formData.cost),
          estimatedDays: formData.estimatedDays.trim(),
          isActive: formData.isActive
        }),
      });

      if (response.ok) {
        const result = await response.json();
        alert('Shipping method created successfully!');
        router.push('/dashboard/shipping/list');
      } else {
        const error = await response.json();
        throw new Error(error.error || 'Failed to create shipping method');
      }
    } catch (error) {
      console.error('Error creating shipping method:', error);
      alert(`Failed to create shipping method: ${error instanceof Error ? error.message : 'Unknown error'}`);
    } finally {
      setLoading(false);
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? (e.target as HTMLInputElement).checked : value
    }));

    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({
        ...prev,
        [name]: ''
      }));
    }
  };

  const handleLogout = () => {
    AuthService.logout();
    router.push('/');
  };

  if (!user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-900">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  const navigationItems = [
    { 
      name: 'Dashboard', 
      icon: '📊', 
      href: '/dashboard', 
      id: 'dashboard',
      current: activeMenuItem === 'dashboard'
    },
    { 
      name: 'Category', 
      icon: '📁', 
      href: '#category', 
      id: 'category',
      current: activeMenuItem === 'category',
      hasSubMenu: true,
      isOpen: categoryMenuOpen,
      subItems: [
        { name: 'Add New Category', href: '/dashboard/category/add-new', id: 'add-category', icon: '➕' },
        { name: 'List of Categories', href: '/dashboard/category/list', id: 'list-categories', icon: '📃' }
      ]
    },
    { 
      name: 'Product', 
      icon: '📦', 
      href: '#product', 
      id: 'product',
      current: activeMenuItem === 'product',
      hasSubMenu: true,
      isOpen: productMenuOpen,
      subItems: [
        { name: 'Add New Product', href: '/dashboard/product/add-new', id: 'add-product', icon: '🛍️' },
        { name: 'List of Products', href: '/dashboard/product/list', id: 'list-products', icon: '📃' }
      ]
    },
    { 
      name: 'Orders', 
      icon: '🛒', 
      href: '#orders', 
      id: 'orders',
      current: activeMenuItem === 'orders',
      hasSubMenu: true,
      isOpen: orderMenuOpen,
      subItems: [
        { name: 'All Orders', href: '/dashboard/orders/list', id: 'all-orders', icon: '📋' },
        { name: 'Pending Orders', href: '/dashboard/orders/pending', id: 'pending-orders', icon: '⏳' },
        { name: 'Completed Orders', href: '/dashboard/orders/completed', id: 'completed-orders', icon: '✅' },
        { name: 'Cancelled Orders', href: '/dashboard/orders/cancelled', id: 'cancelled-orders', icon: '❌' },
        { name: 'Order Analytics', href: '/dashboard/orders/analytics', id: 'order-analytics', icon: '📊' }
      ]
    },
    { 
      name: 'Shipping', 
      icon: '🚚', 
      href: '#shipping', 
      id: 'shipping',
      current: activeMenuItem === 'shipping',
      hasSubMenu: true,
      isOpen: shippingMenuOpen,
      subItems: [
        { name: 'Shipping Methods', href: '/dashboard/shipping/list', id: 'shipping-list', icon: '📋' },
        { name: 'Add New Method', href: '/dashboard/shipping/add-new', id: 'add-shipping', icon: '➕' }
      ]
    },
    { 
      name: 'Analytics', 
      icon: '📈', 
      href: '#analytics', 
      id: 'analytics',
      current: activeMenuItem === 'analytics'
    },
    { 
      name: 'Customers', 
      icon: '👥', 
      href: '#customers', 
      id: 'customers',
      current: activeMenuItem === 'customers'
    },
    { 
      name: 'Settings', 
      icon: '⚙️', 
      href: '#settings', 
      id: 'settings',
      current: activeMenuItem === 'settings'
    },
  ];

  const handleMenuClick = (itemId: string, hasSubMenu?: boolean) => {
    if (hasSubMenu) {
      if (itemId === 'category') {
        setCategoryMenuOpen(!categoryMenuOpen);
      } else if (itemId === 'product') {
        setProductMenuOpen(!productMenuOpen);
      } else if (itemId === 'orders') {
        setOrderMenuOpen(!orderMenuOpen);
      } else if (itemId === 'shipping') {
        setShippingMenuOpen(!shippingMenuOpen);
      }
    } else {
      setActiveMenuItem(itemId);
      // Close any open submenus when clicking other items
      if (itemId !== 'category') setCategoryMenuOpen(false);
      if (itemId !== 'product') setProductMenuOpen(false);
      if (itemId !== 'orders') setOrderMenuOpen(false);
      if (itemId !== 'shipping') setShippingMenuOpen(false);
    }
  };

  const handleSubMenuClick = (subItemId: string, parentId: string, href?: string) => {
    setActiveMenuItem(subItemId);
    // Keep the parent menu open
    if (parentId === 'category') {
      setCategoryMenuOpen(true);
    } else if (parentId === 'product') {
      setProductMenuOpen(true);
    } else if (parentId === 'orders') {
      setOrderMenuOpen(true);
    } else if (parentId === 'shipping') {
      setShippingMenuOpen(true);
    }
    
    // Navigate to actual route if href is provided
    if (href && href.startsWith('/')) {
      router.push(href);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 flex">
      {/* Sidebar */}
      <div className={`${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} fixed inset-y-0 left-0 z-50 w-64 bg-gray-900 transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0`}>
        <div className="flex items-center justify-center h-16 bg-gray-800">
          <h1 className="text-white text-xl font-bold">📱 MyApp</h1>
        </div>
        
        <nav className="mt-8 px-4">
          <ul className="space-y-2">
            {navigationItems.map((item) => (
              <li key={item.name}>
                <div>
                  <button
                    onClick={() => handleMenuClick(item.id, item.hasSubMenu)}
                    className={`group flex items-center justify-between w-full px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
                      item.current || (item.hasSubMenu && item.isOpen)
                        ? 'bg-blue-600 text-white'
                        : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                    }`}
                  >
                    <div className="flex items-center">
                      <span className="mr-3 text-lg">{item.icon}</span>
                      {item.name}
                    </div>
                    {item.hasSubMenu && (
                      <svg 
                        className={`w-4 h-4 transition-transform ${item.isOpen ? 'rotate-180' : ''}`}
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    )}
                  </button>
                  
                  {/* Submenu */}
                  {item.hasSubMenu && item.isOpen && item.subItems && (
                    <ul className="mt-2 space-y-1 pl-4">
                      {item.subItems.map((subItem) => (
                        <li key={subItem.id}>
                          <button
                            onClick={() => handleSubMenuClick(subItem.id, item.id, subItem.href)}
                            className={`group flex items-center w-full px-4 py-2 text-sm rounded-lg transition-colors ${
                              activeMenuItem === subItem.id
                                ? 'bg-blue-500 text-white'
                                : 'text-gray-400 hover:bg-gray-700 hover:text-white'
                            }`}
                          >
                            <span className="mr-3 text-base">{subItem.icon}</span>
                            {subItem.name}
                          </button>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </nav>
        
        <div className="absolute bottom-0 w-full p-4">
          <div className="bg-gray-800 rounded-lg p-4">
            <div className="flex items-center">
              <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center">
                <span className="text-white font-semibold text-sm">
                  {user?.email?.charAt(0).toUpperCase() || 'U'}
                </span>
              </div>
              <div className="ml-3">
                <p className="text-sm font-medium text-white truncate">{user?.email || 'User'}</p>
                <p className="text-xs text-gray-400">Online</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Sidebar overlay for mobile */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 z-40 bg-gray-600 bg-opacity-75 lg:hidden" 
          onClick={() => setSidebarOpen(false)}
        ></div>
      )}

      {/* Main content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Header */}
        <header className="bg-white shadow-sm border-b border-gray-200">
          <div className="flex items-center justify-between px-6 py-4">
            <div className="flex items-center">
              <button
                className="text-gray-500 hover:text-gray-600 lg:hidden"
                onClick={() => setSidebarOpen(true)}
              >
                <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
              <div className="ml-4 lg:ml-0">
                <h1 className="text-2xl font-semibold text-gray-900">Add New Shipping Method</h1>
                <p className="text-sm text-gray-600 mt-1">Create a new shipping method for your store</p>
              </div>
            </div>
            
            <div className="flex items-center space-x-4">
              <Link
                href="/dashboard/shipping/list"
                className="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                ← Back to List
              </Link>
              <button
                onClick={handleLogout}
                className="bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Logout
              </button>
            </div>
          </div>
        </header>

        {/* Main content area */}
        <main className="flex-1 overflow-y-auto bg-gray-50 p-6">
          <div className="max-w-2xl mx-auto">
            <div className="bg-white rounded-lg shadow-sm p-6">
              <form onSubmit={handleSubmit} className="space-y-6">
                {/* Method Name */}
                <div>
                  <label htmlFor="name" className="block text-sm font-medium text-gray-700 mb-2">
                    Method Name *
                  </label>
                  <input
                    type="text"
                    id="name"
                    name="name"
                    value={formData.name}
                    onChange={handleInputChange}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.name ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="e.g., Express Shipping"
                  />
                  {errors.name && (
                    <p className="mt-1 text-sm text-red-600">{errors.name}</p>
                  )}
                </div>

                {/* Description */}
                <div>
                  <label htmlFor="description" className="block text-sm font-medium text-gray-700 mb-2">
                    Description *
                  </label>
                  <textarea
                    id="description"
                    name="description"
                    value={formData.description}
                    onChange={handleInputChange}
                    rows={3}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.description ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="e.g., Fast delivery within 2-3 business days"
                  />
                  {errors.description && (
                    <p className="mt-1 text-sm text-red-600">{errors.description}</p>
                  )}
                </div>

                {/* Cost */}
                <div>
                  <label htmlFor="cost" className="block text-sm font-medium text-gray-700 mb-2">
                    Shipping Cost ($) *
                  </label>
                  <input
                    type="number"
                    id="cost"
                    name="cost"
                    value={formData.cost}
                    onChange={handleInputChange}
                    min="0"
                    step="0.01"
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.cost ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="0.00"
                  />
                  {errors.cost && (
                    <p className="mt-1 text-sm text-red-600">{errors.cost}</p>
                  )}
                  <p className="mt-1 text-sm text-gray-500">Enter 0 for free shipping</p>
                </div>

                {/* Estimated Delivery Time */}
                <div>
                  <label htmlFor="estimatedDays" className="block text-sm font-medium text-gray-700 mb-2">
                    Estimated Delivery Time *
                  </label>
                  <input
                    type="text"
                    id="estimatedDays"
                    name="estimatedDays"
                    value={formData.estimatedDays}
                    onChange={handleInputChange}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.estimatedDays ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="e.g., 2-3 or 1 or 7-10"
                  />
                  {errors.estimatedDays && (
                    <p className="mt-1 text-sm text-red-600">{errors.estimatedDays}</p>
                  )}
                  <p className="mt-1 text-sm text-gray-500">e.g., "1", "2-3", "5-7" (business days)</p>
                </div>

                {/* Active Status */}
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    id="isActive"
                    name="isActive"
                    checked={formData.isActive}
                    onChange={handleInputChange}
                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                  />
                  <label htmlFor="isActive" className="ml-2 block text-sm text-gray-700">
                    Active (available for customers to select)
                  </label>
                </div>

                {/* Form Actions */}
                <div className="flex items-center justify-end space-x-4 pt-6 border-t border-gray-200">
                  <Link
                    href="/dashboard/shipping/list"
                    className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    Cancel
                  </Link>
                  <button
                    type="submit"
                    disabled={loading}
                    className="px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-lg hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed flex items-center gap-2"
                  >
                    {loading ? (
                      <>
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                        Creating...
                      </>
                    ) : (
                      <>
                        <span>➕</span>
                        Create Shipping Method
                      </>
                    )}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </main>
      </div>
    </div>
  );
}