'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import DashboardLayout from '@/components/DashboardLayout';

interface Category {
  id: number;
  $loki?: number;
  name: string;
  slug: string;
  icon: string;
}

interface ProductAttribute {
  id: string;
  name: string;
  values: string[];
}

interface ProductVariant {
  id: string;
  attributes: { [key: string]: string };
  price: number;
  comparePrice?: number;
  stockQuantity: number;
  sku: string;
  isActive: boolean;
}

interface ProductFormData {
  name: string;
  description: string;
  price: string;
  comparePrice: string;
  categoryId: string;
  sku: string;
  barcode: string;
  stockQuantity: string;
  lowStockAlert: string;
  weight: string;
  dimensions: string;
  isActive: boolean;
  isFeatured: boolean;
  tags: string;
  metaTitle: string;
  metaDescription: string;
  type: 'simple' | 'variable';
  attributes: ProductAttribute[];
  variants: ProductVariant[];
}

export default function AddNewProduct() {
  const [categories, setCategories] = useState<Category[]>([]);
  const [formData, setFormData] = useState<ProductFormData>({
    name: '',
    description: '',
    price: '',
    comparePrice: '',
    categoryId: '',
    sku: '',
    barcode: '',
    stockQuantity: '',
    lowStockAlert: '5',
    weight: '',
    dimensions: '',
    isActive: true,
    isFeatured: false,
    tags: '',
    metaTitle: '',
    metaDescription: '',
    type: 'simple',
    attributes: [],
    variants: []
  });
  const [newAttributeName, setNewAttributeName] = useState('');
  const [newAttributeValues, setNewAttributeValues] = useState('');
  interface FormErrors extends Partial<ProductFormData> {
    attributesError?: string;
    variantsError?: string;
  }
  
  const [errors, setErrors] = useState<FormErrors>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitStatus, setSubmitStatus] = useState<'idle' | 'success' | 'error'>('idle');
  const [slug, setSlug] = useState('');
  const router = useRouter();

  useEffect(() => {
    fetchCategories();
  }, []);

  // Auto-generate slug from product name
  useEffect(() => {
    if (formData.name) {
      const generatedSlug = formData.name
        .toLowerCase()
        .replace(/[^a-z0-9]+/g, '-')
        .replace(/(^-|-$)/g, '');
      setSlug(generatedSlug);
    } else {
      setSlug('');
    }
  }, [formData.name]);

  const fetchCategories = async () => {
    try {
      console.log('Fetching categories...');
      const response = await fetch('/api/categories');
      const data = await response.json();
      
      console.log('Categories API response:', data);
      
      if (response.ok) {
        const categoriesList = data.categories || [];
        console.log('Setting categories:', categoriesList);
        setCategories(categoriesList);
      } else {
        console.error('Failed to fetch categories:', data);
      }
    } catch (error) {
      console.error('Error fetching categories:', error);
    }
  };

  const validateForm = (): boolean => {
    const newErrors: FormErrors = {};

    if (!formData.name.trim()) {
      newErrors.name = 'Product name is required';
    } else if (formData.name.length < 2) {
      newErrors.name = 'Product name must be at least 2 characters';
    } else if (formData.name.length > 100) {
      newErrors.name = 'Product name must be less than 100 characters';
    }

    if (!formData.description.trim()) {
      newErrors.description = 'Description is required';
    } else if (formData.description.length < 10) {
      newErrors.description = 'Description must be at least 10 characters';
    }

    // For simple products, validate price and stock
    if (formData.type === 'simple') {
      if (!formData.price.trim()) {
        newErrors.price = 'Price is required';
      } else {
        const price = parseFloat(formData.price);
        if (isNaN(price) || price <= 0) {
          newErrors.price = 'Price must be a valid positive number';
        }
      }

      if (formData.comparePrice && formData.price) {
        const price = parseFloat(formData.price);
        const comparePrice = parseFloat(formData.comparePrice);
        if (!isNaN(comparePrice) && comparePrice <= price) {
          newErrors.comparePrice = 'Compare price must be higher than regular price';
        }
      }

      if (!formData.stockQuantity.trim()) {
        newErrors.stockQuantity = 'Stock quantity is required';
      } else {
        const stock = parseInt(formData.stockQuantity);
        if (isNaN(stock) || stock < 0) {
          newErrors.stockQuantity = 'Stock quantity must be a valid number';
        }
      }

      if (!formData.sku.trim()) {
        newErrors.sku = 'SKU is required';
      } else if (formData.sku.length < 3) {
        newErrors.sku = 'SKU must be at least 3 characters';
      }
    } else {
      // For variable products, ensure we have at least one attribute and one variant
      if (formData.attributes.length === 0) {
        newErrors.attributesError = 'At least one attribute is required for variable products';
      }
      
      if (formData.variants.length === 0) {
        newErrors.variantsError = 'At least one variant is required for variable products';
      }
    }

    if (!formData.categoryId) {
      newErrors.categoryId = 'Category is required';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target;
    
    if (type === 'checkbox') {
      const target = e.target as HTMLInputElement;
      setFormData(prev => ({
        ...prev,
        [name]: target.checked
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
    }

    // Clear error when user starts typing
    if (errors[name as keyof ProductFormData]) {
      setErrors(prev => ({
        ...prev,
        [name]: undefined
      }));
    }

    // Clear variants error when switching product type
    if (name === 'type') {
      setErrors(prev => ({
        ...prev,
        variantsError: undefined,
        attributesError: undefined
      }));
    }
  };

  const addAttribute = () => {
    if (!newAttributeName.trim() || !newAttributeValues.trim()) return;
    
    const values = newAttributeValues.split(',').map(v => v.trim()).filter(v => v);
    if (values.length === 0) return;

    const newAttribute: ProductAttribute = {
      id: Date.now().toString(),
      name: newAttributeName.trim(),
      values
    };

    setFormData(prev => ({
      ...prev,
      attributes: [...prev.attributes, newAttribute]
    }));

    setNewAttributeName('');
    setNewAttributeValues('');
    generateVariants([...formData.attributes, newAttribute]);
  };

  const removeAttribute = (attributeId: string) => {
    const updatedAttributes = formData.attributes.filter(attr => attr.id !== attributeId);
    setFormData(prev => ({
      ...prev,
      attributes: updatedAttributes
    }));
    generateVariants(updatedAttributes);
  };

  const generateVariants = (attributes: ProductAttribute[]) => {
    if (attributes.length === 0) {
      setFormData(prev => ({
        ...prev,
        variants: []
      }));
      return;
    }

    // Generate all possible combinations
    const combinations: { [key: string]: string }[] = [];
    
    const generateCombinations = (index: number, current: { [key: string]: string }) => {
      if (index === attributes.length) {
        combinations.push({ ...current });
        return;
      }
      
      const attribute = attributes[index];
      for (const value of attribute.values) {
        current[attribute.name] = value;
        generateCombinations(index + 1, current);
      }
    };

    generateCombinations(0, {});

    const newVariants: ProductVariant[] = combinations.map((combo, index) => {
      const variantName = Object.entries(combo).map(([key, value]) => `${key}: ${value}`).join(', ');
      
      return {
        id: `${Date.now()}-${index}`,
        attributes: combo,
        price: 0,
        comparePrice: 0,
        stockQuantity: 0,
        sku: `${formData.sku || 'SKU'}-VAR-${index + 1}`,
        isActive: true
      };
    });

    setFormData(prev => ({
      ...prev,
      variants: newVariants
    }));
  };

  const updateVariant = (variantId: string, field: keyof ProductVariant, value: any) => {
    setFormData(prev => ({
      ...prev,
      variants: prev.variants.map(variant =>
        variant.id === variantId
          ? { ...variant, [field]: value }
          : variant
      )
    }));
  };

  const handleProductTypeChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const newType = e.target.value as 'simple' | 'variable';
    setFormData(prev => ({
      ...prev,
      type: newType,
      attributes: newType === 'simple' ? [] : prev.attributes,
      variants: newType === 'simple' ? [] : prev.variants
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    console.log('Form submitted, validating...');
    console.log('Form data:', formData);
    
    const isValid = validateForm();
    console.log('Validation result:', isValid);
    console.log('Validation errors:', errors);
    
    if (!isValid) {
      console.log('Form validation failed, stopping submission');
      return;
    }

    setIsSubmitting(true);
    setSubmitStatus('idle');

    const payload = {
      ...formData,
      slug,
      // For simple products, use form price and stock
      price: formData.type === 'simple' ? parseFloat(formData.price) : undefined,
      comparePrice: formData.type === 'simple' && formData.comparePrice ? parseFloat(formData.comparePrice) : undefined,
      stockQuantity: formData.type === 'simple' ? parseInt(formData.stockQuantity) : undefined,
      sku: formData.type === 'simple' ? formData.sku : undefined,
      // For variable products, calculate price range from variants
      priceRange: formData.type === 'variable' && formData.variants.length > 0 ? {
        min: Math.min(...formData.variants.map(v => v.price)),
        max: Math.max(...formData.variants.map(v => v.price))
      } : undefined,
      categoryId: parseInt(formData.categoryId),
      lowStockAlert: parseInt(formData.lowStockAlert),
      weight: formData.weight ? parseFloat(formData.weight) : null,
      tags: formData.tags ? formData.tags.split(',').map(tag => tag.trim()).filter(tag => tag) : [],
      // Ensure variants have proper data types
      variants: formData.type === 'variable' ? formData.variants.map(variant => ({
        ...variant,
        price: parseFloat(variant.price.toString()) || 0,
        comparePrice: variant.comparePrice ? parseFloat(variant.comparePrice.toString()) || undefined : undefined,
        stockQuantity: parseInt(variant.stockQuantity.toString()) || 0
      })) : []
    };

    console.log('Product form data being submitted:', payload);

    try {
      const response = await fetch('/api/products', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(payload),
      });

      console.log('Response status:', response.status);

      if (response.ok) {
        const result = await response.json();
        console.log('Success result:', result);
        setSubmitStatus('success');
        setTimeout(() => {
          router.push('/dashboard/product/list');
        }, 2000);
      } else {
        const errorData = await response.json();
        console.error('Error response:', errorData);
        setSubmitStatus('error');
      }
    } catch (error) {
      console.error('Network error submitting form:', error);
      setSubmitStatus('error');
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <DashboardLayout 
      title="Add New Product" 
      activeMenuItem="product"
      headerActions={
        <Link
          href="/dashboard/product/list"
          className="bg-gray-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-gray-700 transition-colors"
        >
          View Products
        </Link>
      }
    >
      <div className="max-w-4xl mx-auto">
        {/* Success Message */}
        {submitStatus === 'success' && (
          <div className="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
            <div className="flex items-center">
              <div className="flex-shrink-0">
                <span className="text-green-500 text-xl">✅</span>
              </div>
              <div className="ml-3">
                <h3 className="text-sm font-medium text-green-800">Product created successfully!</h3>
                <div className="mt-1 text-sm text-green-700">
                  <p>Your product has been added to the catalog. Redirecting to product list...</p>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Error Message */}
        {submitStatus === 'error' && (
          <div className="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
            <div className="flex items-center">
              <div className="flex-shrink-0">
                <span className="text-red-500 text-xl">❌</span>
              </div>
              <div className="ml-3">
                <h3 className="text-sm font-medium text-red-800">Error creating product</h3>
                <div className="mt-1 text-sm text-red-700">
                  <p>Please check your information and try again.</p>
                </div>
              </div>
            </div>
          </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-8">
          {/* Basic Information */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-lg font-semibold text-gray-900">Basic Information</h2>
              <p className="text-sm text-gray-600 mt-1">Essential product details</p>
            </div>
            
            <div className="p-6 space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {/* Product Name */}
                <div>
                  <label htmlFor="name" className="block text-sm font-medium text-gray-700 mb-2">
                    Product Name *
                  </label>
                  <input
                    type="text"
                    id="name"
                    name="name"
                    value={formData.name}
                    onChange={handleInputChange}
                    className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors ${
                      errors.name ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                    }`}
                    placeholder="Enter product name"
                  />
                  {errors.name && <p className="mt-1 text-sm text-red-600">{errors.name}</p>}
                </div>

                {/* Category */}
                <div>
                  <label htmlFor="categoryId" className="block text-sm font-medium text-gray-700 mb-2">
                    Category *
                  </label>
                  <select
                    id="categoryId"
                    name="categoryId"
                    value={formData.categoryId}
                    onChange={handleInputChange}
                    className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors ${
                      errors.categoryId ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                    }`}
                  >
                    <option value="">Select a category</option>
                    {categories.map((category) => (
                      <option key={category.$loki || category.id} value={category.$loki || category.id}>
                        {category.icon} {category.name}
                      </option>
                    ))}
                  </select>
                  {errors.categoryId && <p className="mt-1 text-sm text-red-600">{errors.categoryId}</p>}
                </div>
              </div>

              {/* Description */}
              <div>
                <label htmlFor="description" className="block text-sm font-medium text-gray-700 mb-2">
                  Description *
                </label>
                <textarea
                  id="description"
                  name="description"
                  value={formData.description}
                  onChange={handleInputChange}
                  rows={4}
                  className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors resize-none ${
                    errors.description ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                  }`}
                  placeholder="Describe the product..."
                />
                {errors.description && <p className="mt-1 text-sm text-red-600">{errors.description}</p>}
              </div>

              {/* Slug Preview */}
              {slug && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    URL Slug (Auto-generated)
                  </label>
                  <div className="px-4 py-3 bg-gray-50 border border-gray-300 rounded-lg text-sm text-gray-600">
                    /products/{slug}
                  </div>
                </div>
              )}
            </div>
          </div>

          {/* Product Type */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-lg font-semibold text-gray-900">Product Type</h2>
              <p className="text-sm text-gray-600 mt-1">Choose between simple or variable product</p>
            </div>
            
            <div className="p-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <label className={`border-2 rounded-lg p-4 cursor-pointer transition-colors ${
                  formData.type === 'simple' ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'
                }`}>
                  <input
                    type="radio"
                    name="type"
                    value="simple"
                    checked={formData.type === 'simple'}
                    onChange={handleProductTypeChange}
                    className="sr-only"
                  />
                  <div className="flex items-center">
                    <div className={`w-4 h-4 rounded-full border-2 mr-3 ${
                      formData.type === 'simple' ? 'border-blue-500 bg-blue-500' : 'border-gray-300'
                    }`}>
                      {formData.type === 'simple' && (
                        <div className="w-2 h-2 bg-white rounded-full mx-auto mt-0.5"></div>
                      )}
                    </div>
                    <div>
                      <h3 className="font-medium text-gray-900">Simple Product</h3>
                      <p className="text-sm text-gray-600">A product with no variations</p>
                    </div>
                  </div>
                </label>

                <label className={`border-2 rounded-lg p-4 cursor-pointer transition-colors ${
                  formData.type === 'variable' ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'
                }`}>
                  <input
                    type="radio"
                    name="type"
                    value="variable"
                    checked={formData.type === 'variable'}
                    onChange={handleProductTypeChange}
                    className="sr-only"
                  />
                  <div className="flex items-center">
                    <div className={`w-4 h-4 rounded-full border-2 mr-3 ${
                      formData.type === 'variable' ? 'border-blue-500 bg-blue-500' : 'border-gray-300'
                    }`}>
                      {formData.type === 'variable' && (
                        <div className="w-2 h-2 bg-white rounded-full mx-auto mt-0.5"></div>
                      )}
                    </div>
                    <div>
                      <h3 className="font-medium text-gray-900">Variable Product</h3>
                      <p className="text-sm text-gray-600">A product with variations (size, color, etc.)</p>
                    </div>
                  </div>
                </label>
              </div>
            </div>
          </div>

          {/* Attributes & Variants (only for variable products) */}
          {formData.type === 'variable' && (
            <div className="bg-white rounded-xl shadow-sm border border-gray-200">
              <div className="p-6 border-b border-gray-200">
                <h2 className="text-lg font-semibold text-gray-900">Product Attributes</h2>
                <p className="text-sm text-gray-600 mt-1">Define attributes like size, color, material, etc.</p>
              </div>
              
              <div className="p-6 space-y-6">
                {/* Add New Attribute */}
                <div className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                  <h3 className="font-medium text-gray-900 mb-4">Add New Attribute</h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Attribute Name
                      </label>
                      <input
                        type="text"
                        value={newAttributeName}
                        onChange={(e) => setNewAttributeName(e.target.value)}
                        placeholder="e.g., Size, Color, Material"
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Values (comma-separated)
                      </label>
                      <input
                        type="text"
                        value={newAttributeValues}
                        onChange={(e) => setNewAttributeValues(e.target.value)}
                        placeholder="e.g., Small, Medium, Large"
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      />
                    </div>
                    <div className="flex items-end">
                      <button
                        type="button"
                        onClick={addAttribute}
                        disabled={!newAttributeName.trim() || !newAttributeValues.trim()}
                        className="w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                      >
                        Add Attribute
                      </button>
                    </div>
                  </div>
                </div>

                {/* Display Attributes */}
                {formData.attributes.length > 0 && (
                  <div>
                    <h3 className="font-medium text-gray-900 mb-4">Current Attributes</h3>
                    <div className="space-y-3">
                      {formData.attributes.map((attribute) => (
                        <div key={attribute.id} className="border border-gray-200 rounded-lg p-4">
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              <h4 className="font-medium text-gray-900">{attribute.name}</h4>
                              <div className="flex flex-wrap gap-2 mt-2">
                                {attribute.values.map((value) => (
                                  <span
                                    key={value}
                                    className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full"
                                  >
                                    {value}
                                  </span>
                                ))}
                              </div>
                            </div>
                            <button
                              type="button"
                              onClick={() => removeAttribute(attribute.id)}
                              className="text-red-600 hover:text-red-800 ml-4"
                            >
                              <span className="text-lg">×</span>
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                )}

                {/* Attributes Error */}
                {errors.attributesError && (
                  <p className="text-sm text-red-600">{errors.attributesError}</p>
                )}

                {/* Generated Variants */}
                {formData.variants.length > 0 && (
                  <div>
                    <h3 className="font-medium text-gray-900 mb-4">
                      Product Variants ({formData.variants.length})
                    </h3>
                    <div className="space-y-4 max-h-96 overflow-y-auto">
                      {formData.variants.map((variant, index) => (
                        <div key={variant.id} className="border border-gray-200 rounded-lg p-4">
                          <div className="flex items-center justify-between mb-3">
                            <h4 className="font-medium text-gray-900">
                              Variant {index + 1}: {Object.entries(variant.attributes).map(([key, value]) => `${key}: ${value}`).join(', ')}
                            </h4>
                          </div>
                          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Price *
                              </label>
                              <div className="relative">
                                <span className="absolute left-3 top-2 text-gray-500 text-sm">$</span>
                                <input
                                  type="number"
                                  step="0.01"
                                  min="0"
                                  value={variant.price || ''}
                                  onChange={(e) => updateVariant(variant.id, 'price', parseFloat(e.target.value) || 0)}
                                  className="w-full pl-7 pr-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-sm"
                                  placeholder="0.00"
                                />
                              </div>
                            </div>
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Compare Price
                              </label>
                              <div className="relative">
                                <span className="absolute left-3 top-2 text-gray-500 text-sm">$</span>
                                <input
                                  type="number"
                                  step="0.01"
                                  min="0"
                                  value={variant.comparePrice || ''}
                                  onChange={(e) => updateVariant(variant.id, 'comparePrice', parseFloat(e.target.value) || undefined)}
                                  className="w-full pl-7 pr-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-sm"
                                  placeholder="0.00"
                                />
                              </div>
                            </div>
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Stock *
                              </label>
                              <input
                                type="number"
                                min="0"
                                value={variant.stockQuantity || ''}
                                onChange={(e) => updateVariant(variant.id, 'stockQuantity', parseInt(e.target.value) || 0)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-sm"
                                placeholder="0"
                              />
                            </div>
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                SKU *
                              </label>
                              <input
                                type="text"
                                value={variant.sku}
                                onChange={(e) => updateVariant(variant.id, 'sku', e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-sm"
                                placeholder="SKU-VAR-1"
                              />
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                )}

                {/* Variants Error */}
                {errors.variantsError && (
                  <p className="text-sm text-red-600">{errors.variantsError}</p>
                )}
              </div>
            </div>
          )}

          {/* Pricing & Inventory (only for simple products) */}
          {formData.type === 'simple' && (
            <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-lg font-semibold text-gray-900">Pricing & Inventory</h2>
              <p className="text-sm text-gray-600 mt-1">Set prices and manage stock</p>
            </div>
            
            <div className="p-6 space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {/* Price */}
                <div>
                  <label htmlFor="price" className="block text-sm font-medium text-gray-700 mb-2">
                    Price *
                  </label>
                  <div className="relative">
                    <span className="absolute left-3 top-3 text-gray-500">$</span>
                    <input
                      type="number"
                      id="price"
                      name="price"
                      value={formData.price}
                      onChange={handleInputChange}
                      step="0.01"
                      min="0"
                      className={`w-full pl-8 pr-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors ${
                        errors.price ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                      }`}
                      placeholder="0.00"
                    />
                  </div>
                  {errors.price && <p className="mt-1 text-sm text-red-600">{errors.price}</p>}
                </div>

                {/* Compare Price */}
                <div>
                  <label htmlFor="comparePrice" className="block text-sm font-medium text-gray-700 mb-2">
                    Compare Price
                  </label>
                  <div className="relative">
                    <span className="absolute left-3 top-3 text-gray-500">$</span>
                    <input
                      type="number"
                      id="comparePrice"
                      name="comparePrice"
                      value={formData.comparePrice}
                      onChange={handleInputChange}
                      step="0.01"
                      min="0"
                      className={`w-full pl-8 pr-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors ${
                        errors.comparePrice ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                      }`}
                      placeholder="0.00"
                    />
                  </div>
                  {errors.comparePrice && <p className="mt-1 text-sm text-red-600">{errors.comparePrice}</p>}
                </div>

                {/* SKU */}
                <div>
                  <label htmlFor="sku" className="block text-sm font-medium text-gray-700 mb-2">
                    SKU *
                  </label>
                  <input
                    type="text"
                    id="sku"
                    name="sku"
                    value={formData.sku}
                    onChange={handleInputChange}
                    className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors ${
                      errors.sku ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                    }`}
                    placeholder="Enter SKU"
                  />
                  {errors.sku && <p className="mt-1 text-sm text-red-600">{errors.sku}</p>}
                </div>

                {/* Stock Quantity */}
                <div>
                  <label htmlFor="stockQuantity" className="block text-sm font-medium text-gray-700 mb-2">
                    Stock Quantity *
                  </label>
                  <input
                    type="number"
                    id="stockQuantity"
                    name="stockQuantity"
                    value={formData.stockQuantity}
                    onChange={handleInputChange}
                    min="0"
                    className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 transition-colors ${
                      errors.stockQuantity ? 'border-red-300 focus:border-red-500' : 'border-gray-300 focus:border-blue-500'
                    }`}
                    placeholder="0"
                  />
                  {errors.stockQuantity && <p className="mt-1 text-sm text-red-600">{errors.stockQuantity}</p>}
                </div>
              </div>
            </div>
          </div>
          )}

          {/* Settings */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-lg font-semibold text-gray-900">Product Settings</h2>
              <p className="text-sm text-gray-600 mt-1">Configure product visibility and features</p>
            </div>
            
            <div className="p-6 space-y-6">
              <div className="flex items-center justify-between">
                <div>
                  <label className="text-sm font-medium text-gray-900">Active Product</label>
                  <p className="text-sm text-gray-600">Product will be visible in the store</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input
                    type="checkbox"
                    name="isActive"
                    checked={formData.isActive}
                    onChange={handleInputChange}
                    className="sr-only peer"
                  />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <label className="text-sm font-medium text-gray-900">Featured Product</label>
                  <p className="text-sm text-gray-600">Show this product in featured sections</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input
                    type="checkbox"
                    name="isFeatured"
                    checked={formData.isFeatured}
                    onChange={handleInputChange}
                    className="sr-only peer"
                  />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>
            </div>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end">
            <button
              type="submit"
              disabled={isSubmitting}
              className="bg-blue-600 text-white px-8 py-3 rounded-lg text-sm font-medium hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
            >
              {isSubmitting ? 'Creating Product...' : 'Create Product'}
            </button>
          </div>
        </form>
      </div>
    </DashboardLayout>
  );
}