'use client';

import { useEffect, useState } from 'react';
import { AuthService, AuthUser } from '@/lib/auth';
import { useRouter } from 'next/navigation';
import Link from 'next/link';

interface OrderItem {
  id: number;
  name: string;
  slug: string;
  price: number;
  quantity: number;
  comparePrice?: number;
}

interface Order {
  $loki?: number;
  id?: number;
  orderNumber: string;
  customerInfo: {
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
  };
  shippingAddress: {
    address: string;
    city: string;
    state: string;
    zipCode: string;
    country: string;
  };
  items: OrderItem[];
  pricing: {
    subtotal: number;
    shipping: number;
    tax: number;
    codFee: number;
    total: number;
  };
  paymentMethod: 'stripe' | 'paypal' | 'card' | 'cod';
  paymentStatus: 'pending' | 'paid' | 'failed';
  orderStatus: 'pending' | 'processing' | 'shipped' | 'delivered' | 'cancelled';
  createdAt: string;
  updatedAt: string;
}

export default function PendingOrders() {
  const [user, setUser] = useState<AuthUser | null>(null);
  const [loading, setLoading] = useState(true);
  const [orders, setOrders] = useState<Order[]>([]);
  const [filteredOrders, setFilteredOrders] = useState<Order[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [categoryMenuOpen, setCategoryMenuOpen] = useState(false);
  const [productMenuOpen, setProductMenuOpen] = useState(false);
  const [orderMenuOpen, setOrderMenuOpen] = useState(true);
  const [activeMenuItem, setActiveMenuItem] = useState('pending-orders');
  const router = useRouter();

  useEffect(() => {
    const currentUser = AuthService.getUser();
    if (!currentUser) {
      router.push('/login');
      return;
    }
    setUser(currentUser);
    fetchPendingOrders();
  }, [router]);

  const fetchPendingOrders = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/orders?status=pending');
      if (response.ok) {
        const ordersData = await response.json();
        setOrders(ordersData);
        setFilteredOrders(ordersData);
      } else {
        console.error('Failed to fetch orders');
        setOrders([]);
        setFilteredOrders([]);
      }
    } catch (error) {
      console.error('Error fetching orders:', error);
      setOrders([]);
      setFilteredOrders([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    let filtered = orders;

    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(order =>
        order.orderNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
        order.customerInfo.firstName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        order.customerInfo.lastName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        order.customerInfo.email.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    setFilteredOrders(filtered);
  }, [orders, searchTerm]);

  const handleLogout = () => {
    AuthService.logout();
    router.push('/');
  };

  const getPaymentMethodIcon = (method: string) => {
    switch (method) {
      case 'stripe': return '💳';
      case 'paypal': return '🏦';
      case 'card': return '💎';
      case 'cod': return '💰';
      default: return '💳';
    }
  };

  const getPaymentMethodName = (method: string) => {
    switch (method) {
      case 'stripe': return 'Stripe';
      case 'paypal': return 'PayPal';
      case 'card': return 'Credit Card';
      case 'cod': return 'Cash on Delivery';
      default: return method;
    }
  };

  const updateOrderStatus = async (orderId: number, newStatus: string) => {
    try {
      // This would be implemented with a PATCH endpoint
      console.log(`Updating order ${orderId} to status: ${newStatus}`);
      // For now, just refresh the orders
      fetchPendingOrders();
    } catch (error) {
      console.error('Error updating order status:', error);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-900">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  const navigationItems = [
    { 
      name: 'Dashboard', 
      icon: '📊', 
      href: '/dashboard', 
      id: 'dashboard',
      current: activeMenuItem === 'dashboard'
    },
    { 
      name: 'Category', 
      icon: '📁', 
      href: '#category', 
      id: 'category',
      current: activeMenuItem === 'category',
      hasSubMenu: true,
      isOpen: categoryMenuOpen,
      subItems: [
        { name: 'Add New Category', href: '/dashboard/category/add-new', id: 'add-category', icon: '➕' },
        { name: 'List of Categories', href: '/dashboard/category/list', id: 'list-categories', icon: '📃' }
      ]
    },
    { 
      name: 'Product', 
      icon: '📦', 
      href: '#product', 
      id: 'product',
      current: activeMenuItem === 'product',
      hasSubMenu: true,
      isOpen: productMenuOpen,
      subItems: [
        { name: 'Add New Product', href: '/dashboard/product/add-new', id: 'add-product', icon: '🛍️' },
        { name: 'List of Products', href: '/dashboard/product/list', id: 'list-products', icon: '📃' }
      ]
    },
    { 
      name: 'Analytics', 
      icon: '📈', 
      href: '#analytics', 
      id: 'analytics',
      current: activeMenuItem === 'analytics'
    },
    { 
      name: 'Orders', 
      icon: '🛒', 
      href: '#orders', 
      id: 'orders',
      current: activeMenuItem === 'orders',
      hasSubMenu: true,
      isOpen: orderMenuOpen,
      subItems: [
        { name: 'All Orders', href: '/dashboard/orders/list', id: 'all-orders', icon: '📋' },
        { name: 'Pending Orders', href: '/dashboard/orders/pending', id: 'pending-orders', icon: '⏳' },
        { name: 'Completed Orders', href: '/dashboard/orders/completed', id: 'completed-orders', icon: '✅' },
        { name: 'Cancelled Orders', href: '/dashboard/orders/cancelled', id: 'cancelled-orders', icon: '❌' },
        { name: 'Order Analytics', href: '/dashboard/orders/analytics', id: 'order-analytics', icon: '📊' }
      ]
    },
    { 
      name: 'Customers', 
      icon: '👥', 
      href: '#customers', 
      id: 'customers',
      current: activeMenuItem === 'customers'
    },
    { 
      name: 'Settings', 
      icon: '⚙️', 
      href: '#settings', 
      id: 'settings',
      current: activeMenuItem === 'settings'
    },
  ];

  const handleMenuClick = (itemId: string, hasSubMenu?: boolean) => {
    if (hasSubMenu) {
      if (itemId === 'category') {
        setCategoryMenuOpen(!categoryMenuOpen);
      } else if (itemId === 'product') {
        setProductMenuOpen(!productMenuOpen);
      } else if (itemId === 'orders') {
        setOrderMenuOpen(!orderMenuOpen);
      }
    } else {
      setActiveMenuItem(itemId);
      // Close any open submenus when clicking other items
      if (itemId !== 'category') setCategoryMenuOpen(false);
      if (itemId !== 'product') setProductMenuOpen(false);
      if (itemId !== 'orders') setOrderMenuOpen(false);
    }
  };

  const handleSubMenuClick = (subItemId: string, parentId: string, href?: string) => {
    setActiveMenuItem(subItemId);
    // Keep the parent menu open
    if (parentId === 'category') {
      setCategoryMenuOpen(true);
    } else if (parentId === 'product') {
      setProductMenuOpen(true);
    } else if (parentId === 'orders') {
      setOrderMenuOpen(true);
    }
    
    // Navigate to actual route if href is provided
    if (href && href.startsWith('/')) {
      router.push(href);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 flex">
      {/* Sidebar - Same as dashboard */}
      <div className={`${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} fixed inset-y-0 left-0 z-50 w-64 bg-gray-900 transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0`}>
        <div className="flex items-center justify-center h-16 bg-gray-800">
          <h1 className="text-white text-xl font-bold">📱 MyApp</h1>
        </div>
        
        <nav className="mt-8 px-4">
          <ul className="space-y-2">
            {navigationItems.map((item) => (
              <li key={item.name}>
                <div>
                  <button
                    onClick={() => handleMenuClick(item.id, item.hasSubMenu)}
                    className={`group flex items-center justify-between w-full px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
                      item.current || (item.hasSubMenu && item.isOpen)
                        ? 'bg-blue-600 text-white'
                        : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                    }`}
                  >
                    <div className="flex items-center">
                      <span className="mr-3 text-lg">{item.icon}</span>
                      {item.name}
                    </div>
                    {item.hasSubMenu && (
                      <svg 
                        className={`w-4 h-4 transition-transform ${item.isOpen ? 'rotate-180' : ''}`}
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    )}
                  </button>
                  
                  {/* Submenu */}
                  {item.hasSubMenu && item.isOpen && item.subItems && (
                    <ul className="mt-2 space-y-1 pl-4">
                      {item.subItems.map((subItem) => (
                        <li key={subItem.id}>
                          <button
                            onClick={() => handleSubMenuClick(subItem.id, item.id, subItem.href)}
                            className={`group flex items-center w-full px-4 py-2 text-sm rounded-lg transition-colors ${
                              activeMenuItem === subItem.id
                                ? 'bg-blue-500 text-white'
                                : 'text-gray-400 hover:bg-gray-700 hover:text-white'
                            }`}
                          >
                            <span className="mr-3 text-base">{subItem.icon}</span>
                            {subItem.name}
                          </button>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </nav>
        
        <div className="absolute bottom-0 w-full p-4">
          <div className="bg-gray-800 rounded-lg p-4">
            <div className="flex items-center">
              <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center">
                <span className="text-white font-semibold text-sm">
                  {user?.email?.charAt(0).toUpperCase() || 'U'}
                </span>
              </div>
              <div className="ml-3">
                <p className="text-sm font-medium text-white truncate">{user?.email || 'User'}</p>
                <p className="text-xs text-gray-400">Online</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Sidebar overlay for mobile */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 z-40 bg-gray-600 bg-opacity-75 lg:hidden" 
          onClick={() => setSidebarOpen(false)}
        ></div>
      )}

      {/* Main content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Header */}
        <header className="bg-white shadow-sm border-b border-gray-200">
          <div className="flex items-center justify-between px-6 py-4">
            <div className="flex items-center">
              <button
                className="text-gray-500 hover:text-gray-600 lg:hidden"
                onClick={() => setSidebarOpen(true)}
              >
                <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
              <h1 className="ml-4 lg:ml-0 text-2xl font-semibold text-gray-900">Pending Orders</h1>
            </div>
            
            <div className="flex items-center space-x-4">
              <button
                onClick={fetchPendingOrders}
                className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Refresh
              </button>
              <button
                onClick={handleLogout}
                className="bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Logout
              </button>
            </div>
          </div>
        </header>

        {/* Main content area */}
        <main className="flex-1 overflow-y-auto bg-gray-50 p-6">
          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-yellow-100">
                  <span className="text-2xl">⏳</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Pending Orders</p>
                  <p className="text-2xl font-semibold text-gray-900">{orders.length}</p>
                </div>
              </div>
            </div>
            
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-green-100">
                  <span className="text-2xl">💰</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">COD Orders</p>
                  <p className="text-2xl font-semibold text-gray-900">
                    {orders.filter(order => order.paymentMethod === 'cod').length}
                  </p>
                </div>
              </div>
            </div>
            
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-blue-100">
                  <span className="text-2xl">💵</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Value</p>
                  <p className="text-2xl font-semibold text-gray-900">
                    ${orders.reduce((sum, order) => sum + order.pricing.total, 0).toFixed(2)}
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Search */}
          <div className="mb-6 bg-white rounded-lg shadow-sm p-6">
            <div className="relative">
              <input
                type="text"
                placeholder="Search pending orders..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
              <div className="absolute inset-y-0 left-0 pl-3 flex items-center">
                <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                </svg>
              </div>
            </div>
          </div>

          {/* Orders List */}
          <div className="bg-white rounded-lg shadow-sm overflow-hidden">
            {filteredOrders.length === 0 ? (
              <div className="text-center py-12">
                <div className="text-4xl mb-4">⏳</div>
                <h3 className="text-lg font-medium text-gray-900 mb-2">No pending orders</h3>
                <p className="text-gray-500">
                  {searchTerm 
                    ? 'No orders match your search criteria'
                    : 'New orders will appear here when customers place them'
                  }
                </p>
              </div>
            ) : (
              <div className="divide-y divide-gray-200">
                {filteredOrders.map((order) => (
                  <div key={order.$loki || order.id} className="p-6 hover:bg-gray-50">
                    <div className="flex items-center justify-between mb-4">
                      <div>
                        <h3 className="text-lg font-semibold text-gray-900">{order.orderNumber}</h3>
                        <p className="text-sm text-gray-500">
                          {new Date(order.createdAt).toLocaleDateString()} at {new Date(order.createdAt).toLocaleTimeString()}
                        </p>
                      </div>
                      <div className="flex items-center space-x-2">
                        <span className="inline-flex px-2 py-1 text-xs font-medium rounded-full bg-yellow-100 text-yellow-800">
                          Pending
                        </span>
                        <div className="flex items-center">
                          <span className="text-lg mr-2">{getPaymentMethodIcon(order.paymentMethod)}</span>
                          <span className="text-sm text-gray-600">{getPaymentMethodName(order.paymentMethod)}</span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                      {/* Customer Info */}
                      <div>
                        <h4 className="font-medium text-gray-900 mb-2">Customer</h4>
                        <p className="text-sm text-gray-600">
                          {order.customerInfo.firstName} {order.customerInfo.lastName}
                        </p>
                        <p className="text-sm text-gray-600">{order.customerInfo.email}</p>
                        <p className="text-sm text-gray-600">{order.customerInfo.phone}</p>
                      </div>
                      
                      {/* Shipping Address */}
                      <div>
                        <h4 className="font-medium text-gray-900 mb-2">Shipping Address</h4>
                        <p className="text-sm text-gray-600">{order.shippingAddress.address}</p>
                        <p className="text-sm text-gray-600">
                          {order.shippingAddress.city}, {order.shippingAddress.state} {order.shippingAddress.zipCode}
                        </p>
                        <p className="text-sm text-gray-600">{order.shippingAddress.country}</p>
                      </div>
                      
                      {/* Order Summary */}
                      <div>
                        <h4 className="font-medium text-gray-900 mb-2">Order Summary</h4>
                        <p className="text-sm text-gray-600">{order.items.length} item{order.items.length !== 1 ? 's' : ''}</p>
                        <p className="text-sm text-gray-600">Subtotal: ${order.pricing.subtotal.toFixed(2)}</p>
                        {order.pricing.codFee > 0 && (
                          <p className="text-sm text-gray-600">COD Fee: ${order.pricing.codFee.toFixed(2)}</p>
                        )}
                        <p className="text-sm font-semibold text-gray-900">Total: ${order.pricing.total.toFixed(2)}</p>
                      </div>
                    </div>

                    {/* Items */}
                    <div className="mt-4">
                      <h4 className="font-medium text-gray-900 mb-2">Items</h4>
                      <div className="space-y-2">
                        {order.items.map((item, index) => (
                          <div key={index} className="flex justify-between items-center text-sm">
                            <span className="text-gray-900">{item.name} x {item.quantity}</span>
                            <span className="text-gray-600">${(item.price * item.quantity).toFixed(2)}</span>
                          </div>
                        ))}
                      </div>
                    </div>

                    {/* Actions */}
                    <div className="mt-4 flex space-x-3">
                      <button
                        onClick={() => updateOrderStatus(order.$loki || order.id || 0, 'processing')}
                        className="px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700"
                      >
                        Mark as Processing
                      </button>
                      <button
                        onClick={() => updateOrderStatus(order.$loki || order.id || 0, 'cancelled')}
                        className="px-4 py-2 bg-red-600 text-white text-sm font-medium rounded-md hover:bg-red-700"
                      >
                        Cancel Order
                      </button>
                      <button className="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-md hover:bg-gray-200">
                        View Details
                      </button>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </main>
      </div>
    </div>
  );
}