'use client';

import { useEffect, useState } from 'react';
import { AuthService, AuthUser } from '@/lib/auth';
import { useRouter } from 'next/navigation';
import Link from 'next/link';

interface OrderItem {
  id: number;
  name: string;
  slug: string;
  price: number;
  quantity: number;
  comparePrice?: number;
}

interface CustomerInfo {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
}

interface ShippingAddress {
  address: string;
  city: string;
  state: string;
  zipCode: string;
  country: string;
}

interface Pricing {
  subtotal: number;
  shipping: number;
  tax: number;
  codFee?: number;
  total: number;
}

interface Order {
  id?: number;
  orderNumber: string;
  customerInfo: CustomerInfo;
  shippingAddress: ShippingAddress;
  items: OrderItem[];
  pricing: Pricing;
  paymentMethod: 'stripe' | 'paypal' | 'card' | 'cod';
  orderStatus: 'pending' | 'processing' | 'shipped' | 'delivered' | 'cancelled';
  paymentStatus: 'pending' | 'paid' | 'failed';
  createdAt: string;
}

export default function OrdersList() {
  const [user, setUser] = useState<AuthUser | null>(null);
  const [loading, setLoading] = useState(true);
  const [orders, setOrders] = useState<Order[]>([]);
  const [filteredOrders, setFilteredOrders] = useState<Order[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [updating, setUpdating] = useState<string | null>(null);
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [categoryMenuOpen, setCategoryMenuOpen] = useState(false);
  const [productMenuOpen, setProductMenuOpen] = useState(false);
  const [orderMenuOpen, setOrderMenuOpen] = useState(true);
  const [shippingMenuOpen, setShippingMenuOpen] = useState(false);
  const [customersMenuOpen, setCustomersMenuOpen] = useState(false);
  const [activeMenuItem, setActiveMenuItem] = useState('all-orders');
  const router = useRouter();

  useEffect(() => {
    const currentUser = AuthService.getUser();
    if (!currentUser) {
      router.push('/login');
      return;
    }
    setUser(currentUser);
    fetchOrders();
  }, [router]);

  const fetchOrders = async () => {
    try {
      setLoading(true);
      console.log('📡 Fetching orders from API...');
      const response = await fetch('/api/orders');
      console.log('📥 Orders API response status:', response.status);
      
      if (response.ok) {
        const ordersData = await response.json();
        console.log('✅ Orders fetched successfully:', ordersData.length, 'orders');
        setOrders(ordersData);
        setFilteredOrders(ordersData);
      } else {
        console.error('❌ Failed to fetch orders:', response.status);
        setOrders([]);
        setFilteredOrders([]);
      }
    } catch (error) {
      console.error('❌ Error fetching orders:', error);
      setOrders([]);
      setFilteredOrders([]);
    } finally {
      setLoading(false);
    }
  };

  const updateOrderStatus = async (orderNumber: string, newStatus: string) => {
    console.log(`🔄 Attempting to update order ${orderNumber} to ${newStatus}`);
    
    try {
      setUpdating(orderNumber);
      
      console.log(`📡 Making API call to /api/orders/${orderNumber}`);
      const response = await fetch(`/api/orders/${orderNumber}`, {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ orderStatus: newStatus }),
      });

      console.log(`📥 API Response status: ${response.status}`);

      if (response.ok) {
        const result = await response.json();
        console.log('✅ Order updated successfully:', result);
        
        // Update local state immediately
        const updatedOrders = orders.map(order => 
          order.orderNumber === orderNumber 
            ? { ...order, orderStatus: newStatus as any, updatedAt: new Date().toISOString() }
            : order
        );
        
        console.log('🔄 Updating local state...');
        setOrders(updatedOrders);
        
        // Re-apply filters with updated orders
        console.log('🔄 Applying filters...');
        applyFilters(updatedOrders);
        
        // Refresh orders from database to ensure consistency
        console.log('🔄 Scheduling database refresh...');
        setTimeout(() => {
          console.log('🔄 Refreshing from database...');
          fetchOrders();
        }, 1000);
        
        // Show success message
        console.log('✅ Status update completed successfully');
        alert(`✅ Order ${orderNumber} status updated to ${newStatus}`);
      } else {
        const error = await response.text();
        console.error('❌ API Error:', error);
        throw new Error(error || 'Failed to update order');
      }
    } catch (error) {
      console.error('❌ Error updating order status:', error);
      alert(`❌ Failed to update order status: ${error instanceof Error ? error.message : 'Unknown error'}`);
    } finally {
      console.log('🏁 Update process finished');
      setUpdating(null);
    }
  };

  const applyFilters = (ordersToFilter: Order[]) => {
    let filtered = ordersToFilter;

    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(order =>
        order.orderNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
        `${order.customerInfo.firstName} ${order.customerInfo.lastName}`.toLowerCase().includes(searchTerm.toLowerCase()) ||
        order.customerInfo.email.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    // Filter by status
    if (statusFilter !== 'all') {
      filtered = filtered.filter(order => order.orderStatus === statusFilter);
    }

    setFilteredOrders(filtered);
  };

  useEffect(() => {
    applyFilters(orders);
  }, [orders, searchTerm, statusFilter]);

  const handleLogout = () => {
    AuthService.logout();
    router.push('/');
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'processing': return 'bg-blue-100 text-blue-800';
      case 'shipped': return 'bg-purple-100 text-purple-800';
      case 'delivered': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPaymentMethodIcon = (method: string) => {
    switch (method) {
      case 'stripe': return '💳';
      case 'paypal': return '🏦';
      case 'card': return '💎';
      case 'cod': return '💰';
      default: return '💳';
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-900">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  const navigationItems = [
    { 
      name: 'Dashboard', 
      icon: '📊', 
      href: '/dashboard', 
      id: 'dashboard',
      current: activeMenuItem === 'dashboard'
    },
    { 
      name: 'Category', 
      icon: '📁', 
      href: '#category', 
      id: 'category',
      current: activeMenuItem === 'category',
      hasSubMenu: true,
      isOpen: categoryMenuOpen,
      subItems: [
        { name: 'Add New Category', href: '/dashboard/category/add-new', id: 'add-category', icon: '➕' },
        { name: 'List of Categories', href: '/dashboard/category/list', id: 'list-categories', icon: '📃' }
      ]
    },
    { 
      name: 'Product', 
      icon: '📦', 
      href: '#product', 
      id: 'product',
      current: activeMenuItem === 'product',
      hasSubMenu: true,
      isOpen: productMenuOpen,
      subItems: [
        { name: 'Add New Product', href: '/dashboard/product/add-new', id: 'add-product', icon: '🛍️' },
        { name: 'List of Products', href: '/dashboard/product/list', id: 'list-products', icon: '📃' }
      ]
    },
    { 
      name: 'Analytics', 
      icon: '📈', 
      href: '#analytics', 
      id: 'analytics',
      current: activeMenuItem === 'analytics'
    },
    { 
      name: 'Orders', 
      icon: '🛒', 
      href: '#orders', 
      id: 'orders',
      current: activeMenuItem === 'orders',
      hasSubMenu: true,
      isOpen: orderMenuOpen,
      subItems: [
        { name: 'All Orders', href: '/dashboard/orders/list', id: 'all-orders', icon: '📋' },
        { name: 'Pending Orders', href: '/dashboard/orders/pending', id: 'pending-orders', icon: '⏳' },
        { name: 'Completed Orders', href: '/dashboard/orders/completed', id: 'completed-orders', icon: '✅' },
        { name: 'Cancelled Orders', href: '/dashboard/orders/cancelled', id: 'cancelled-orders', icon: '❌' },
        { name: 'Order Analytics', href: '/dashboard/orders/analytics', id: 'order-analytics', icon: '📊' }
      ]
    },
    { 
      name: 'Shipping', 
      icon: '🚚', 
      href: '#shipping', 
      id: 'shipping',
      current: activeMenuItem === 'shipping',
      hasSubMenu: true,
      isOpen: shippingMenuOpen,
      subItems: [
        { name: 'Shipping Methods', href: '/dashboard/shipping/list', id: 'shipping-list', icon: '📋' },
        { name: 'Add New Method', href: '/dashboard/shipping/add-new', id: 'add-shipping', icon: '➕' }
      ]
    },
    { 
      name: 'Customers', 
      icon: '👥', 
      href: '#customers', 
      id: 'customers',
      current: activeMenuItem === 'customers',
      hasSubMenu: true,
      isOpen: customersMenuOpen,
      subItems: [
        { name: 'All Customers', href: '/dashboard/customers/list', id: 'all-customers', icon: '👥' },
        { name: 'Add New Customer', href: '/dashboard/customers/add-new', id: 'add-customer', icon: '➕' }
      ]
    },
    { 
      name: 'Settings', 
      icon: '⚙️', 
      href: '#settings', 
      id: 'settings',
      current: activeMenuItem === 'settings'
    },
  ];

  const handleMenuClick = (itemId: string, hasSubMenu?: boolean) => {
    if (hasSubMenu) {
      if (itemId === 'category') {
        setCategoryMenuOpen(!categoryMenuOpen);
      } else if (itemId === 'product') {
        setProductMenuOpen(!productMenuOpen);
      } else if (itemId === 'orders') {
        setOrderMenuOpen(!orderMenuOpen);
      } else if (itemId === 'shipping') {
        setShippingMenuOpen(!shippingMenuOpen);
      } else if (itemId === 'customers') {
        setCustomersMenuOpen(!customersMenuOpen);
      }
    } else {
      setActiveMenuItem(itemId);
      // Close any open submenus when clicking other items
      if (itemId !== 'category') setCategoryMenuOpen(false);
      if (itemId !== 'product') setProductMenuOpen(false);
      if (itemId !== 'orders') setOrderMenuOpen(false);
      if (itemId !== 'shipping') setShippingMenuOpen(false);
      if (itemId !== 'customers') setCustomersMenuOpen(false);
    }
  };

  const handleSubMenuClick = (subItemId: string, parentId: string, href?: string) => {
    setActiveMenuItem(subItemId);
    // Keep the parent menu open
    if (parentId === 'category') {
      setCategoryMenuOpen(true);
    } else if (parentId === 'product') {
      setProductMenuOpen(true);
    } else if (parentId === 'orders') {
      setOrderMenuOpen(true);
    } else if (parentId === 'shipping') {
      setShippingMenuOpen(true);
    }
    
    // Navigate to actual route if href is provided
    if (href && href.startsWith('/')) {
      router.push(href);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 flex">
      {/* Sidebar - Same as dashboard */}
      <div className={`${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} fixed inset-y-0 left-0 z-50 w-64 bg-gray-900 transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0`}>
        <div className="flex items-center justify-center h-16 bg-gray-800">
          <h1 className="text-white text-xl font-bold">📱 MyApp</h1>
        </div>
        
        <nav className="mt-8 px-4">
          <ul className="space-y-2">
            {navigationItems.map((item) => (
              <li key={item.name}>
                <div>
                  <button
                    onClick={() => handleMenuClick(item.id, item.hasSubMenu)}
                    className={`group flex items-center justify-between w-full px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
                      item.current || (item.hasSubMenu && item.isOpen)
                        ? 'bg-blue-600 text-white'
                        : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                    }`}
                  >
                    <div className="flex items-center">
                      <span className="mr-3 text-lg">{item.icon}</span>
                      {item.name}
                    </div>
                    {item.hasSubMenu && (
                      <svg 
                        className={`w-4 h-4 transition-transform ${item.isOpen ? 'rotate-180' : ''}`}
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    )}
                  </button>
                  
                  {/* Submenu */}
                  {item.hasSubMenu && item.isOpen && item.subItems && (
                    <ul className="mt-2 space-y-1 pl-4">
                      {item.subItems.map((subItem) => (
                        <li key={subItem.id}>
                          <button
                            onClick={() => handleSubMenuClick(subItem.id, item.id, subItem.href)}
                            className={`group flex items-center w-full px-4 py-2 text-sm rounded-lg transition-colors ${
                              activeMenuItem === subItem.id
                                ? 'bg-blue-500 text-white'
                                : 'text-gray-400 hover:bg-gray-700 hover:text-white'
                            }`}
                          >
                            <span className="mr-3 text-base">{subItem.icon}</span>
                            {subItem.name}
                          </button>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </nav>
        
        <div className="absolute bottom-0 w-full p-4">
          <div className="bg-gray-800 rounded-lg p-4">
            <div className="flex items-center">
              <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center">
                <span className="text-white font-semibold text-sm">
                  {user?.email?.charAt(0).toUpperCase() || 'U'}
                </span>
              </div>
              <div className="ml-3">
                <p className="text-sm font-medium text-white truncate">{user?.email || 'User'}</p>
                <p className="text-xs text-gray-400">Online</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Sidebar overlay for mobile */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 z-40 bg-gray-600 bg-opacity-75 lg:hidden" 
          onClick={() => setSidebarOpen(false)}
        ></div>
      )}

      {/* Main content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Header */}
        <header className="bg-white shadow-sm border-b border-gray-200">
          <div className="flex items-center justify-between px-6 py-4">
            <div className="flex items-center">
              <button
                className="text-gray-500 hover:text-gray-600 lg:hidden"
                onClick={() => setSidebarOpen(true)}
              >
                <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
              <h1 className="ml-4 lg:ml-0 text-2xl font-semibold text-gray-900">All Orders</h1>
            </div>
            
            <div className="flex items-center space-x-4">
              <button
                onClick={handleLogout}
                className="bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Logout
              </button>
            </div>
          </div>
        </header>

        {/* Main content area */}
        <main className="flex-1 overflow-y-auto bg-gray-50 p-6">
          {/* Filters and Search */}
          <div className="mb-6 bg-white rounded-lg shadow-sm p-6">
            <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
              <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
                <div className="relative">
                  <input
                    type="text"
                    placeholder="Search orders..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full sm:w-64 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center">
                    <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                    </svg>
                  </div>
                </div>
                
                <select
                  value={statusFilter}
                  onChange={(e) => setStatusFilter(e.target.value)}
                  className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="all">All Status</option>
                  <option value="pending">Pending</option>
                  <option value="processing">Processing</option>
                  <option value="shipped">Shipped</option>
                  <option value="delivered">Delivered</option>
                  <option value="cancelled">Cancelled</option>
                </select>
              </div>
              
              <div className="text-sm text-gray-600">
                Showing {filteredOrders.length} of {orders.length} orders
              </div>
            </div>
          </div>

          {/* Orders Table */}
          <div className="bg-white rounded-lg shadow-sm overflow-hidden">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order Details
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customer
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Payment
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Total
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Date
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {filteredOrders.map((order) => (
                    <tr key={order.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">
                            {order.orderNumber}
                          </div>
                          <div className="text-sm text-gray-500">
                            {order.items.length} item{order.items.length !== 1 ? 's' : ''}
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">
                            {order.customerInfo.firstName} {order.customerInfo.lastName}
                          </div>
                          <div className="text-sm text-gray-500">
                            {order.customerInfo.email}
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(order.orderStatus || 'pending')}`}>
                          {order.orderStatus ? order.orderStatus.charAt(0).toUpperCase() + order.orderStatus.slice(1) : 'Pending'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <span className="text-lg mr-2">{getPaymentMethodIcon(order.paymentMethod)}</span>
                          <span className="text-sm text-gray-900 capitalize">
                            {order.paymentMethod === 'cod' ? 'Cash on Delivery' : order.paymentMethod}
                          </span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                        ${order.pricing.total.toFixed(2)}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {new Date(order.createdAt).toLocaleDateString()}
                      </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                          <div className="flex flex-col space-y-2">
                            <Link
                              href={`/dashboard/orders/${order.orderNumber}`}
                              className="text-blue-600 hover:text-blue-900 text-xs"
                            >
                              View Details
                            </Link>
                            
                            {/* Status change buttons */}
                            <div className="flex flex-wrap gap-1">
                              {order.orderStatus === 'pending' && (
                                <>
                                  <button
                                    onClick={(e) => {
                                      e.preventDefault();
                                      console.log(`🖱️ Process button clicked for order: ${order.orderNumber}`);
                                      updateOrderStatus(order.orderNumber, 'processing');
                                    }}
                                    disabled={updating === order.orderNumber}
                                    className="px-2 py-1 bg-blue-600 text-white text-xs rounded hover:bg-blue-700 disabled:opacity-50"
                                  >
                                    {updating === order.orderNumber ? 'Processing...' : 'Process'}
                                  </button>
                                  <button
                                    onClick={(e) => {
                                      e.preventDefault();
                                      console.log(`🖱️ Cancel button clicked for order: ${order.orderNumber}`);
                                      updateOrderStatus(order.orderNumber, 'cancelled');
                                    }}
                                    disabled={updating === order.orderNumber}
                                    className="px-2 py-1 bg-red-600 text-white text-xs rounded hover:bg-red-700 disabled:opacity-50"
                                  >
                                    {updating === order.orderNumber ? 'Cancelling...' : 'Cancel'}
                                  </button>
                                </>
                              )}
                              {order.orderStatus === 'processing' && (
                                <>
                                  <button
                                    onClick={(e) => {
                                      e.preventDefault();
                                      console.log(`🖱️ Ship button clicked for order: ${order.orderNumber}`);
                                      updateOrderStatus(order.orderNumber, 'shipped');
                                    }}
                                    disabled={updating === order.orderNumber}
                                    className="px-2 py-1 bg-purple-600 text-white text-xs rounded hover:bg-purple-700 disabled:opacity-50"
                                  >
                                    {updating === order.orderNumber ? 'Shipping...' : 'Ship'}
                                  </button>
                                  <button
                                    onClick={(e) => {
                                      e.preventDefault();
                                      console.log(`🖱️ Cancel button clicked for order: ${order.orderNumber}`);
                                      updateOrderStatus(order.orderNumber, 'cancelled');
                                    }}
                                    disabled={updating === order.orderNumber}
                                    className="px-2 py-1 bg-red-600 text-white text-xs rounded hover:bg-red-700 disabled:opacity-50"
                                  >
                                    {updating === order.orderNumber ? 'Cancelling...' : 'Cancel'}
                                  </button>
                                </>
                              )}
                              {order.orderStatus === 'shipped' && (
                                <button
                                  onClick={(e) => {
                                    e.preventDefault();
                                    console.log(`🖱️ Delivered button clicked for order: ${order.orderNumber}`);
                                    updateOrderStatus(order.orderNumber, 'delivered');
                                  }}
                                  disabled={updating === order.orderNumber}
                                  className="px-2 py-1 bg-green-600 text-white text-xs rounded hover:bg-green-700 disabled:opacity-50"
                                >
                                  {updating === order.orderNumber ? 'Delivering...' : 'Delivered'}
                                </button>
                              )}
                              {order.orderStatus === 'cancelled' && (
                                <button
                                  onClick={(e) => {
                                    e.preventDefault();
                                    console.log(`🖱️ Restore button clicked for order: ${order.orderNumber}`);
                                    updateOrderStatus(order.orderNumber, 'pending');
                                  }}
                                  disabled={updating === order.orderNumber}
                                  className="px-2 py-1 bg-yellow-600 text-white text-xs rounded hover:bg-yellow-700 disabled:opacity-50"
                                >
                                  {updating === order.orderNumber ? 'Restoring...' : 'Restore'}
                                </button>
                              )}
                              {order.orderStatus === 'delivered' && (
                                <span className="text-xs text-green-600 font-medium">Completed</span>
                              )}
                            </div>
                          </div>
                        </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            
            {filteredOrders.length === 0 && (
              <div className="text-center py-12">
                <div className="text-4xl mb-4">📋</div>
                <h3 className="text-lg font-medium text-gray-900 mb-2">No orders found</h3>
                <p className="text-gray-500">
                  {searchTerm || statusFilter !== 'all' 
                    ? 'Try adjusting your search criteria'
                    : 'Orders will appear here once customers start placing orders'
                  }
                </p>
              </div>
            )}
          </div>
        </main>
      </div>
    </div>
  );
}