'use client';

import { useEffect, useState } from 'react';
import { AuthService, AuthUser } from '@/lib/auth';
import { useRouter } from 'next/navigation';

interface OrderItem {
  id: number;
  name: string;
  slug: string;
  price: number;
  quantity: number;
  comparePrice?: number;
}

interface Order {
  $loki?: number;
  id?: number;
  orderNumber: string;
  customerInfo: {
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
  };
  shippingAddress: {
    address: string;
    city: string;
    state: string;
    zipCode: string;
    country: string;
  };
  items: OrderItem[];
  pricing: {
    subtotal: number;
    shipping: number;
    tax: number;
    codFee?: number;
    total: number;
  };
  paymentMethod: 'stripe' | 'paypal' | 'card' | 'cod';
  paymentStatus: 'pending' | 'paid' | 'failed';
  orderStatus: 'pending' | 'processing' | 'shipped' | 'delivered' | 'cancelled';
  createdAt: string;
  updatedAt: string;
}

interface Analytics {
  totalOrders: number;
  totalRevenue: number;
  averageOrderValue: number;
  conversionRate: number;
  ordersByStatus: {
    pending: number;
    processing: number;
    shipped: number;
    delivered: number;
    cancelled: number;
  };
  ordersByPaymentMethod: {
    stripe: number;
    paypal: number;
    card: number;
    cod: number;
  };
  revenueByPaymentMethod: {
    stripe: number;
    paypal: number;
    card: number;
    cod: number;
  };
  topProducts: Array<{
    name: string;
    quantity: number;
    revenue: number;
  }>;
  monthlyTrends: Array<{
    month: string;
    orders: number;
    revenue: number;
  }>;
}

export default function OrderAnalytics() {
  const [user, setUser] = useState<AuthUser | null>(null);
  const [loading, setLoading] = useState(true);
  const [orders, setOrders] = useState<Order[]>([]);
  const [analytics, setAnalytics] = useState<Analytics | null>(null);
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [categoryMenuOpen, setCategoryMenuOpen] = useState(false);
  const [productMenuOpen, setProductMenuOpen] = useState(false);
  const [orderMenuOpen, setOrderMenuOpen] = useState(true);
  const [activeMenuItem, setActiveMenuItem] = useState('order-analytics');
  const router = useRouter();

  useEffect(() => {
    const currentUser = AuthService.getUser();
    if (!currentUser) {
      router.push('/login');
      return;
    }
    setUser(currentUser);
    fetchOrdersAndAnalytics();
  }, [router]);

  const fetchOrdersAndAnalytics = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/orders');
      if (response.ok) {
        const ordersData = await response.json();
        setOrders(ordersData);
        calculateAnalytics(ordersData);
      } else {
        console.error('Failed to fetch orders');
      }
    } catch (error) {
      console.error('Error fetching orders:', error);
    } finally {
      setLoading(false);
    }
  };

  const calculateAnalytics = (ordersData: Order[]) => {
    const totalOrders = ordersData.length;
    const totalRevenue = ordersData.reduce((sum, order) => sum + order.pricing.total, 0);
    const averageOrderValue = totalRevenue / totalOrders || 0;

    // Calculate orders by status
    const ordersByStatus = {
      pending: ordersData.filter(order => order.orderStatus === 'pending').length,
      processing: ordersData.filter(order => order.orderStatus === 'processing').length,
      shipped: ordersData.filter(order => order.orderStatus === 'shipped').length,
      delivered: ordersData.filter(order => order.orderStatus === 'delivered').length,
      cancelled: ordersData.filter(order => order.orderStatus === 'cancelled').length,
    };

    // Calculate orders by payment method
    const ordersByPaymentMethod = {
      stripe: ordersData.filter(order => order.paymentMethod === 'stripe').length,
      paypal: ordersData.filter(order => order.paymentMethod === 'paypal').length,
      card: ordersData.filter(order => order.paymentMethod === 'card').length,
      cod: ordersData.filter(order => order.paymentMethod === 'cod').length,
    };

    // Calculate revenue by payment method
    const revenueByPaymentMethod = {
      stripe: ordersData.filter(order => order.paymentMethod === 'stripe').reduce((sum, order) => sum + order.pricing.total, 0),
      paypal: ordersData.filter(order => order.paymentMethod === 'paypal').reduce((sum, order) => sum + order.pricing.total, 0),
      card: ordersData.filter(order => order.paymentMethod === 'card').reduce((sum, order) => sum + order.pricing.total, 0),
      cod: ordersData.filter(order => order.paymentMethod === 'cod').reduce((sum, order) => sum + order.pricing.total, 0),
    };

    // Calculate top products
    const productMap = new Map<string, { quantity: number; revenue: number }>();
    ordersData.forEach(order => {
      order.items.forEach(item => {
        const existing = productMap.get(item.name) || { quantity: 0, revenue: 0 };
        productMap.set(item.name, {
          quantity: existing.quantity + item.quantity,
          revenue: existing.revenue + (item.price * item.quantity)
        });
      });
    });

    const topProducts = Array.from(productMap.entries())
      .map(([name, data]) => ({ name, ...data }))
      .sort((a, b) => b.revenue - a.revenue)
      .slice(0, 10);

    // Calculate monthly trends (last 12 months)
    const monthlyTrends = [];
    for (let i = 11; i >= 0; i--) {
      const date = new Date();
      date.setMonth(date.getMonth() - i);
      const monthStart = new Date(date.getFullYear(), date.getMonth(), 1);
      const monthEnd = new Date(date.getFullYear(), date.getMonth() + 1, 0);
      
      const monthOrders = ordersData.filter(order => {
        const orderDate = new Date(order.createdAt);
        return orderDate >= monthStart && orderDate <= monthEnd;
      });

      monthlyTrends.push({
        month: date.toLocaleDateString('en-US', { month: 'short', year: 'numeric' }),
        orders: monthOrders.length,
        revenue: monthOrders.reduce((sum, order) => sum + order.pricing.total, 0)
      });
    }

    const analyticsData: Analytics = {
      totalOrders,
      totalRevenue,
      averageOrderValue,
      conversionRate: 85.5, // This would be calculated based on site visits vs orders
      ordersByStatus,
      ordersByPaymentMethod,
      revenueByPaymentMethod,
      topProducts,
      monthlyTrends
    };

    setAnalytics(analyticsData);
  };

  const handleLogout = () => {
    AuthService.logout();
    router.push('/');
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-900">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  if (!user || !analytics) {
    return null;
  }

  const navigationItems = [
    { 
      name: 'Dashboard', 
      icon: '📊', 
      href: '/dashboard', 
      id: 'dashboard',
      current: activeMenuItem === 'dashboard'
    },
    { 
      name: 'Category', 
      icon: '📁', 
      href: '#category', 
      id: 'category',
      current: activeMenuItem === 'category',
      hasSubMenu: true,
      isOpen: categoryMenuOpen,
      subItems: [
        { name: 'Add New Category', href: '/dashboard/category/add-new', id: 'add-category', icon: '➕' },
        { name: 'List of Categories', href: '/dashboard/category/list', id: 'list-categories', icon: '📃' }
      ]
    },
    { 
      name: 'Product', 
      icon: '📦', 
      href: '#product', 
      id: 'product',
      current: activeMenuItem === 'product',
      hasSubMenu: true,
      isOpen: productMenuOpen,
      subItems: [
        { name: 'Add New Product', href: '/dashboard/product/add-new', id: 'add-product', icon: '🛍️' },
        { name: 'List of Products', href: '/dashboard/product/list', id: 'list-products', icon: '📃' }
      ]
    },
    { 
      name: 'Analytics', 
      icon: '📈', 
      href: '#analytics', 
      id: 'analytics',
      current: activeMenuItem === 'analytics'
    },
    { 
      name: 'Orders', 
      icon: '🛒', 
      href: '#orders', 
      id: 'orders',
      current: activeMenuItem === 'orders',
      hasSubMenu: true,
      isOpen: orderMenuOpen,
      subItems: [
        { name: 'All Orders', href: '/dashboard/orders/list', id: 'all-orders', icon: '📋' },
        { name: 'Pending Orders', href: '/dashboard/orders/pending', id: 'pending-orders', icon: '⏳' },
        { name: 'Completed Orders', href: '/dashboard/orders/completed', id: 'completed-orders', icon: '✅' },
        { name: 'Cancelled Orders', href: '/dashboard/orders/cancelled', id: 'cancelled-orders', icon: '❌' },
        { name: 'Order Analytics', href: '/dashboard/orders/analytics', id: 'order-analytics', icon: '📊' }
      ]
    },
    { 
      name: 'Customers', 
      icon: '👥', 
      href: '#customers', 
      id: 'customers',
      current: activeMenuItem === 'customers'
    },
    { 
      name: 'Settings', 
      icon: '⚙️', 
      href: '#settings', 
      id: 'settings',
      current: activeMenuItem === 'settings'
    },
  ];

  const handleMenuClick = (itemId: string, hasSubMenu?: boolean) => {
    if (hasSubMenu) {
      if (itemId === 'category') {
        setCategoryMenuOpen(!categoryMenuOpen);
      } else if (itemId === 'product') {
        setProductMenuOpen(!productMenuOpen);
      } else if (itemId === 'orders') {
        setOrderMenuOpen(!orderMenuOpen);
      }
    } else {
      setActiveMenuItem(itemId);
    }
  };

  const handleSubMenuClick = (subItemId: string, parentId: string, href?: string) => {
    setActiveMenuItem(subItemId);
    if (parentId === 'orders') {
      setOrderMenuOpen(true);
    }
    
    if (href && href.startsWith('/')) {
      router.push(href);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 flex">
      {/* Sidebar */}
      <div className={`${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} fixed inset-y-0 left-0 z-50 w-64 bg-gray-900 transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0`}>
        <div className="flex items-center justify-center h-16 bg-gray-800">
          <h1 className="text-white text-xl font-bold">📱 MyApp</h1>
        </div>
        
        <nav className="mt-8 px-4">
          <ul className="space-y-2">
            {navigationItems.map((item) => (
              <li key={item.name}>
                <div>
                  <button
                    onClick={() => handleMenuClick(item.id, item.hasSubMenu)}
                    className={`group flex items-center justify-between w-full px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
                      item.current || (item.hasSubMenu && item.isOpen)
                        ? 'bg-blue-600 text-white'
                        : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                    }`}
                  >
                    <div className="flex items-center">
                      <span className="mr-3 text-lg">{item.icon}</span>
                      {item.name}
                    </div>
                    {item.hasSubMenu && (
                      <svg 
                        className={`w-4 h-4 transition-transform ${item.isOpen ? 'rotate-180' : ''}`}
                        fill="none" 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                      </svg>
                    )}
                  </button>
                  
                  {item.hasSubMenu && item.isOpen && item.subItems && (
                    <ul className="mt-2 space-y-1 pl-4">
                      {item.subItems.map((subItem) => (
                        <li key={subItem.id}>
                          <button
                            onClick={() => handleSubMenuClick(subItem.id, item.id, subItem.href)}
                            className={`group flex items-center w-full px-4 py-2 text-sm rounded-lg transition-colors ${
                              activeMenuItem === subItem.id
                                ? 'bg-blue-500 text-white'
                                : 'text-gray-400 hover:bg-gray-700 hover:text-white'
                            }`}
                          >
                            <span className="mr-3 text-base">{subItem.icon}</span>
                            {subItem.name}
                          </button>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              </li>
            ))}
          </ul>
        </nav>
        
        <div className="absolute bottom-0 w-full p-4">
          <div className="bg-gray-800 rounded-lg p-4">
            <div className="flex items-center">
              <div className="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center">
                <span className="text-white font-semibold text-sm">
                  {user?.email?.charAt(0).toUpperCase() || 'U'}
                </span>
              </div>
              <div className="ml-3">
                <p className="text-sm font-medium text-white truncate">{user?.email || 'User'}</p>
                <p className="text-xs text-gray-400">Online</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Sidebar overlay for mobile */}
      {sidebarOpen && (
        <div 
          className="fixed inset-0 z-40 bg-gray-600 bg-opacity-75 lg:hidden" 
          onClick={() => setSidebarOpen(false)}
        ></div>
      )}

      {/* Main content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Header */}
        <header className="bg-white shadow-sm border-b border-gray-200">
          <div className="flex items-center justify-between px-6 py-4">
            <div className="flex items-center">
              <button
                className="text-gray-500 hover:text-gray-600 lg:hidden"
                onClick={() => setSidebarOpen(true)}
              >
                <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                </svg>
              </button>
              <h1 className="ml-4 lg:ml-0 text-2xl font-semibold text-gray-900">Order Analytics</h1>
            </div>
            
            <div className="flex items-center space-x-4">
              <button
                onClick={fetchOrdersAndAnalytics}
                className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Refresh Data
              </button>
              <button
                onClick={handleLogout}
                className="bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors"
              >
                Logout
              </button>
            </div>
          </div>
        </header>

        {/* Main content area */}
        <main className="flex-1 overflow-y-auto bg-gray-50 p-6">
          {/* Key Metrics */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-blue-100">
                  <span className="text-2xl">📋</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Orders</p>
                  <p className="text-2xl font-semibold text-gray-900">{analytics.totalOrders}</p>
                </div>
              </div>
            </div>
            
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-green-100">
                  <span className="text-2xl">💰</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Revenue</p>
                  <p className="text-2xl font-semibold text-gray-900">${analytics.totalRevenue.toFixed(2)}</p>
                </div>
              </div>
            </div>
            
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-purple-100">
                  <span className="text-2xl">📊</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Avg Order Value</p>
                  <p className="text-2xl font-semibold text-gray-900">${analytics.averageOrderValue.toFixed(2)}</p>
                </div>
              </div>
            </div>
            
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center">
                <div className="p-3 rounded-full bg-orange-100">
                  <span className="text-2xl">🎯</span>
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Conversion Rate</p>
                  <p className="text-2xl font-semibold text-gray-900">{analytics.conversionRate}%</p>
                </div>
              </div>
            </div>
          </div>

          {/* Charts Row */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            {/* Order Status Distribution */}
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Orders by Status</h3>
              <div className="space-y-4">
                {Object.entries(analytics.ordersByStatus).map(([status, count]) => (
                  <div key={status} className="flex items-center justify-between">
                    <div className="flex items-center">
                      <div className={`w-4 h-4 rounded mr-3 ${
                        status === 'pending' ? 'bg-yellow-400' :
                        status === 'processing' ? 'bg-blue-400' :
                        status === 'shipped' ? 'bg-purple-400' :
                        status === 'delivered' ? 'bg-green-400' :
                        'bg-red-400'
                      }`}></div>
                      <span className="text-sm font-medium text-gray-700 capitalize">{status}</span>
                    </div>
                    <div className="flex items-center">
                      <span className="text-sm font-semibold text-gray-900 mr-4">{count}</span>
                      <div className="w-20 bg-gray-200 rounded-full h-2">
                        <div 
                          className={`h-2 rounded-full ${
                            status === 'pending' ? 'bg-yellow-400' :
                            status === 'processing' ? 'bg-blue-400' :
                            status === 'shipped' ? 'bg-purple-400' :
                            status === 'delivered' ? 'bg-green-400' :
                            'bg-red-400'
                          }`}
                          style={{ width: `${(count / analytics.totalOrders) * 100}%` }}
                        ></div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Payment Methods */}
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Methods</h3>
              <div className="space-y-4">
                {Object.entries(analytics.ordersByPaymentMethod).map(([method, count]) => (
                  <div key={method} className="flex items-center justify-between">
                    <div className="flex items-center">
                      <span className="text-lg mr-3">
                        {method === 'stripe' ? '💳' : 
                         method === 'paypal' ? '🏦' : 
                         method === 'card' ? '💎' : '💰'}
                      </span>
                      <span className="text-sm font-medium text-gray-700 capitalize">
                        {method === 'cod' ? 'Cash on Delivery' : method}
                      </span>
                    </div>
                    <div className="flex items-center">
                      <span className="text-sm font-semibold text-gray-900 mr-2">{count}</span>
                      <span className="text-sm text-gray-500">
                        (${analytics.revenueByPaymentMethod[method as keyof typeof analytics.revenueByPaymentMethod].toFixed(2)})
                      </span>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* Top Products & Monthly Trends */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            {/* Top Products */}
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Top Products</h3>
              <div className="space-y-4">
                {analytics.topProducts.slice(0, 5).map((product, index) => (
                  <div key={product.name} className="flex items-center justify-between">
                    <div className="flex items-center">
                      <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center mr-3">
                        <span className="text-sm font-semibold text-blue-600">#{index + 1}</span>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-900">{product.name}</p>
                        <p className="text-xs text-gray-500">{product.quantity} units sold</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-semibold text-gray-900">${product.revenue.toFixed(2)}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Monthly Trends */}
            <div className="bg-white rounded-lg shadow-sm p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Monthly Trends</h3>
              <div className="space-y-4">
                {analytics.monthlyTrends.slice(-6).map((month) => (
                  <div key={month.month} className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium text-gray-900">{month.month}</p>
                      <p className="text-xs text-gray-500">{month.orders} orders</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-semibold text-gray-900">${month.revenue.toFixed(2)}</p>
                      <div className="w-20 bg-gray-200 rounded-full h-2 mt-1">
                        <div 
                          className="h-2 rounded-full bg-blue-400"
                          style={{ 
                            width: `${Math.max(10, (month.revenue / Math.max(...analytics.monthlyTrends.map(m => m.revenue))) * 100)}%` 
                          }}
                        ></div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </main>
      </div>
    </div>
  );
}