'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useCart } from '../../context/CartContext';
// Simple SVG icons as components
const ShoppingCartIcon = ({ className }: { className?: string }) => (
  <svg className={className} fill="none" stroke="currentColor" viewBox="0 0 24 24">
    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 3h2l.4 2M7 13h10l4-8H5.4m0 0L4 3H2m5 10v6a1 1 0 001 1h6a1 1 0 001-1v-6m-8 0V9a1 1 0 011-1h6a1 1 0 011 1v4.01" />
  </svg>
);

const TrashIcon = ({ className }: { className?: string }) => (
  <svg className={className} fill="none" stroke="currentColor" viewBox="0 0 24 24">
    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
  </svg>
);

const PlusIcon = ({ className }: { className?: string }) => (
  <svg className={className} fill="none" stroke="currentColor" viewBox="0 0 24 24">
    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
  </svg>
);

const MinusIcon = ({ className }: { className?: string }) => (
  <svg className={className} fill="none" stroke="currentColor" viewBox="0 0 24 24">
    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20 12H4" />
  </svg>
);

interface CartItem {
  id: number;
  name: string;
  slug: string;
  price: number;
  comparePrice?: number;
  quantity: number;
  image?: string;
  variantId?: string;
  variantSku?: string;
  variantAttributes?: { [key: string]: string };
}

interface ShippingMethod {
  id: string;
  name: string;
  description: string;
  cost: number;
  estimatedDays: string;
  isActive: boolean;
}

export default function CartPage() {
  const { items: cartItems, itemCount, totalPrice, updateQuantity, removeItem, clearCart } = useCart();
  const [isLoading, setIsLoading] = useState(false);
  const [shippingMethods, setShippingMethods] = useState<ShippingMethod[]>([]);
  const [selectedShippingMethod, setSelectedShippingMethod] = useState<ShippingMethod | null>(null);
  const [loadingShipping, setLoadingShipping] = useState(false);

  const subtotal = totalPrice;
  const shippingCost = selectedShippingMethod ? selectedShippingMethod.cost : 0;
  const tax = subtotal * 0.1; // 10% tax
  const total = subtotal + shippingCost + tax;

  // Fetch shipping methods
  useEffect(() => {
    const fetchShippingMethods = async () => {
      try {
        setLoadingShipping(true);
        const response = await fetch('/api/shipping');
        if (response.ok) {
          const methods: ShippingMethod[] = await response.json();
          const activeMethods = methods.filter(method => method.isActive);
          setShippingMethods(activeMethods);
          
          // Auto-select free shipping if available, otherwise first method
          const freeShipping = activeMethods.find(method => method.cost === 0);
          const defaultMethod = freeShipping || activeMethods[0];
          if (defaultMethod) {
            setSelectedShippingMethod(defaultMethod);
          }
        } else {
          console.error('Failed to fetch shipping methods');
        }
      } catch (error) {
        console.error('Error fetching shipping methods:', error);
      } finally {
        setLoadingShipping(false);
      }
    };

    if (cartItems.length > 0) {
      fetchShippingMethods();
    }
  }, [cartItems.length]);

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <Link href="/" className="text-2xl font-bold text-blue-600">
              EcomStore
            </Link>
            <div className="flex items-center space-x-4">
              <ShoppingCartIcon className="h-6 w-6" />
              <span className="text-sm text-gray-600">
                {itemCount} items
              </span>
            </div>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="flex items-center justify-between mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Shopping Cart</h1>
          {cartItems.length > 0 && (
            <button
              onClick={clearCart}
              className="text-red-600 hover:text-red-800 text-sm font-medium"
            >
              Clear Cart
            </button>
          )}
        </div>

        {cartItems.length === 0 ? (
          <div className="text-center py-16">
            <ShoppingCartIcon className="mx-auto h-24 w-24 text-gray-300" />
            <h2 className="mt-4 text-2xl font-semibold text-gray-900">Your cart is empty</h2>
            <p className="mt-2 text-gray-600">Start shopping to add items to your cart</p>
            <Link
              href="/"
              className="mt-6 inline-flex items-center px-6 py-3 border border-transparent text-base font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700"
            >
              Continue Shopping
            </Link>
          </div>
        ) : (
          <div className="lg:grid lg:grid-cols-12 lg:gap-x-12 lg:items-start xl:gap-x-16">
            {/* Left Column - Cart Items and Shipping */}
            <div className="lg:col-span-7 space-y-8">
              {/* Cart Items */}
              <div className="bg-white shadow rounded-lg">
                <div className="px-4 py-6 sm:px-6">
                  <h2 className="text-lg font-medium text-gray-900 mb-4">Cart Items</h2>
                  <div className="flow-root">
                    <ul role="list" className="-my-6 divide-y divide-gray-200">
                      {cartItems.map((item) => (
                        <li key={item.id} className="py-6 flex">
                          <div className="flex-shrink-0 w-24 h-24 border border-gray-200 rounded-md overflow-hidden">
                            <div className="w-full h-full bg-gray-200 flex items-center justify-center">
                              <span className="text-gray-500 text-xs">No Image</span>
                            </div>
                          </div>

                          <div className="ml-4 flex-1 flex flex-col">
                            <div>
                              <div className="flex justify-between text-base font-medium text-gray-900">
                                <h3>
                                  <Link href={`/product/${item.slug}`} className="hover:text-blue-600">
                                    {item.name}
                                  </Link>
                                </h3>
                                <p className="ml-4">${item.price.toFixed(2)}</p>
                              </div>
                              {item.comparePrice && item.comparePrice > item.price && (
                                <p className="text-sm text-gray-500 line-through">
                                  ${item.comparePrice.toFixed(2)}
                                </p>
                              )}
                            </div>
                            <div className="flex-1 flex items-end justify-between text-sm">
                              <div className="flex items-center">
                                <button
                                  onClick={() => updateQuantity(item.id, item.quantity - 1, item.variantId)}
                                  className="flex items-center justify-center w-8 h-8 rounded-md border border-gray-300 hover:bg-gray-50"
                                >
                                  <MinusIcon className="h-4 w-4" />
                                </button>
                                <span className="mx-3 font-medium">{item.quantity}</span>
                                <button
                                  onClick={() => updateQuantity(item.id, item.quantity + 1, item.variantId)}
                                  className="flex items-center justify-center w-8 h-8 rounded-md border border-gray-300 hover:bg-gray-50"
                                >
                                  <PlusIcon className="h-4 w-4" />
                                </button>
                              </div>

                              <button
                                onClick={() => removeItem(item.id, item.variantId)}
                                className="font-medium text-red-600 hover:text-red-500 flex items-center"
                              >
                                <TrashIcon className="h-4 w-4 mr-1" />
                                Remove
                              </button>
                            </div>
                          </div>
                        </li>
                      ))}
                    </ul>
                  </div>
                </div>
              </div>

              {/* Shipping Methods */}
              <div className="bg-white shadow rounded-lg">
                <div className="px-4 py-6 sm:px-6">
                  <h2 className="text-lg font-medium text-gray-900 mb-4">Shipping Options</h2>
                  {loadingShipping ? (
                    <div className="flex items-center justify-center py-4">
                      <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
                      <span className="ml-2 text-sm text-gray-600">Loading shipping methods...</span>
                    </div>
                  ) : shippingMethods.length > 0 ? (
                    <div className="space-y-4">
                      {shippingMethods.map((method) => (
                        <label
                          key={method.id}
                          className={`relative flex cursor-pointer rounded-lg border p-4 focus:outline-none ${
                            selectedShippingMethod?.id === method.id
                              ? 'border-blue-600 ring-2 ring-blue-600 bg-blue-50'
                              : 'border-gray-300 hover:border-gray-400'
                          }`}
                        >
                          <input
                            type="radio"
                            name="shipping-method"
                            value={method.id}
                            checked={selectedShippingMethod?.id === method.id}
                            onChange={() => setSelectedShippingMethod(method)}
                            className="sr-only"
                          />
                          <div className="flex w-full justify-between">
                            <div className="flex flex-col">
                              <span className="block text-sm font-medium text-gray-900">
                                {method.name}
                              </span>
                              <span className="block text-sm text-gray-500">
                                {method.description}
                              </span>
                              <span className="block text-xs text-gray-400 mt-1">
                                Delivery: {method.estimatedDays} business days
                              </span>
                            </div>
                            <div className="flex flex-col items-end">
                              <span className="text-sm font-medium text-gray-900">
                                {method.cost === 0 ? 'Free' : `$${method.cost.toFixed(2)}`}
                              </span>
                              {method.cost === 0 && (
                                <span className="text-xs text-green-600 font-medium">
                                  FREE
                                </span>
                              )}
                            </div>
                          </div>
                          {selectedShippingMethod?.id === method.id && (
                            <div className="absolute -inset-px rounded-lg border-2 border-blue-600 pointer-events-none" />
                          )}
                        </label>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-4">
                      <p className="text-sm text-gray-500">No shipping methods available</p>
                    </div>
                  )}
                </div>
              </div>
            </div>

            {/* Right Column - Order Summary */}
            <div className="mt-16 rounded-lg bg-white shadow lg:mt-0 lg:col-span-5">
              <div className="px-4 py-6 sm:px-6">
                <h2 className="text-lg font-medium text-gray-900">Order Summary</h2>
                <div className="mt-6 space-y-4">
                  <div className="flex items-center justify-between">
                    <dt className="text-sm text-gray-600">Subtotal</dt>
                    <dd className="text-sm font-medium text-gray-900">${subtotal.toFixed(2)}</dd>
                  </div>
                  <div className="flex items-center justify-between border-t border-gray-200 pt-4">
                    <div className="flex flex-col">
                      <dt className="text-sm text-gray-600">Shipping</dt>
                      {selectedShippingMethod && (
                        <dd className="text-xs text-gray-500">
                          {selectedShippingMethod.name} ({selectedShippingMethod.estimatedDays} days)
                        </dd>
                      )}
                    </div>
                    <dd className="text-sm font-medium text-gray-900">
                      {shippingCost === 0 ? 'Free' : `$${shippingCost.toFixed(2)}`}
                    </dd>
                  </div>
                  {selectedShippingMethod && shippingCost === 0 && (
                    <p className="text-xs text-green-600">Free shipping selected!</p>
                  )}
                  <div className="flex items-center justify-between">
                    <dt className="text-sm text-gray-600">Tax</dt>
                    <dd className="text-sm font-medium text-gray-900">${tax.toFixed(2)}</dd>
                  </div>
                  <div className="flex items-center justify-between border-t border-gray-200 pt-4">
                    <dt className="text-base font-medium text-gray-900">Total</dt>
                    <dd className="text-base font-medium text-gray-900">${total.toFixed(2)}</dd>
                  </div>
                </div>

                <div className="mt-6">
                  <Link
                    href={`/checkout${selectedShippingMethod ? `?shipping=${selectedShippingMethod.id}` : ''}`}
                    className={`block w-full border border-transparent rounded-md shadow-sm py-3 px-4 text-base font-medium text-white focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 text-center ${
                      selectedShippingMethod 
                        ? 'bg-blue-600 hover:bg-blue-700' 
                        : 'bg-gray-400 cursor-not-allowed'
                    }`}
                    onClick={(e) => {
                      if (!selectedShippingMethod) {
                        e.preventDefault();
                        alert('Please select a shipping method to continue.');
                      }
                    }}
                  >
                    Proceed to Checkout
                  </Link>
                  {!selectedShippingMethod && (
                    <p className="text-xs text-red-600 mt-2 text-center">
                      Please select a shipping method to continue
                    </p>
                  )}
                </div>

                <div className="mt-6 text-center">
                  <Link
                    href="/"
                    className="text-sm font-medium text-blue-600 hover:text-blue-500"
                  >
                    Continue Shopping
                    <span aria-hidden="true"> &rarr;</span>
                  </Link>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}