import { NextRequest, NextResponse } from 'next/server';
import { getDatabase } from '../../../lib/database';

export interface Subcategory {
  id: string;
  name: string;
  description: string;
  slug: string;
  categoryId: string;
  icon: string;
  isActive: boolean;
  sortOrder: number;
  createdAt: string;
  updatedAt: string;
}

export async function GET(request: NextRequest) {
  try {
    const db = await getDatabase();
    let subcategories = db.getCollection<Subcategory>('subcategories');
    let categories = db.getCollection('categories');

    if (!subcategories) {
      // Create empty collection if it doesn't exist
      subcategories = db.addCollection<Subcategory>('subcategories');
    }

    if (!categories) {
      categories = db.addCollection('categories');
    }

    const allSubcategories = subcategories.find({});
    
    // Enhance subcategories with parent category information
    const enhancedSubcategories = allSubcategories.map(subcategory => {
      // Find parent category by matching string ID with categoryId
      const parentCategory = categories.findOne({ 
        $or: [
          { id: subcategory.categoryId },
          { $loki: parseInt(subcategory.categoryId) }
        ]
      });
      
      return {
        ...subcategory,
        categoryId: subcategory.categoryId.toString(), // Ensure categoryId is always a string
        parentCategory: parentCategory ? {
          id: parentCategory.id || parentCategory.$loki.toString(),
          name: parentCategory.name,
          icon: parentCategory.icon,
          slug: parentCategory.slug,
          isActive: parentCategory.isActive
        } : null
      };
    });
    
    // Sort by sortOrder and name
    enhancedSubcategories.sort((a, b) => {
      if (a.sortOrder !== b.sortOrder) {
        return a.sortOrder - b.sortOrder;
      }
      return a.name.localeCompare(b.name);
    });

    return NextResponse.json({
      success: true,
      subcategories: enhancedSubcategories
    });

  } catch (error) {
    console.error('Error fetching subcategories:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { name, description, slug, categoryId, icon, isActive = true, sortOrder = 0 } = body;

    // Validate required fields
    if (!name || !description || !categoryId) {
      return NextResponse.json(
        { error: 'Name, description, and category are required' },
        { status: 400 }
      );
    }

    const db = await getDatabase();
    let subcategories = db.getCollection<Subcategory>('subcategories');

    if (!subcategories) {
      subcategories = db.addCollection<Subcategory>('subcategories');
    }

    // Generate slug if not provided
    const finalSlug = slug || name.toLowerCase()
      .replace(/[^a-z0-9 -]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim();

    // Check if slug already exists
    const existingSubcategory = subcategories.findOne({ slug: finalSlug });
    if (existingSubcategory) {
      return NextResponse.json(
        { error: 'A subcategory with this slug already exists' },
        { status: 409 }
      );
    }

    // Generate unique ID
    const id = `SUBCAT-${Date.now()}-${Math.random().toString(36).substr(2, 9).toUpperCase()}`;

    const newSubcategory: Subcategory = {
      id,
      name: name.trim(),
      description: description.trim(),
      slug: finalSlug,
      categoryId,
      icon: icon || '📁',
      isActive,
      sortOrder,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };

    const insertedSubcategory = subcategories.insert(newSubcategory);
    
    // Save the database
    await db.saveDatabase();

    return NextResponse.json({
      success: true,
      subcategory: insertedSubcategory
    }, { status: 201 });

  } catch (error) {
    console.error('Error creating subcategory:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}