import { NextRequest, NextResponse } from 'next/server';
import Loki from 'lokijs';
import path from 'path';

const dbPath = path.join(process.cwd(), 'users.db');

function initializeDatabase(): Promise<Loki> {
  return new Promise((resolve, reject) => {
    const db = new Loki(dbPath, {
      persistenceMethod: 'fs',
      autoload: true,
      autoloadCallback: (err) => {
        if (err) {
          console.error('Error loading database:', err);
          reject(err);
        } else {
          resolve(db);
        }
      }
    });
  });
}

// GET - Fetch all shipping methods
export async function GET() {
  try {
    const db = await initializeDatabase();
    let shippingCollection = db.getCollection('shipping_methods');

    if (!shippingCollection) {
      // Create collection with default shipping methods
      shippingCollection = db.addCollection('shipping_methods');
      
      const defaultMethods = [
        {
          id: 1,
          name: 'Standard Shipping',
          description: 'Regular delivery within 5-7 business days',
          cost: 5.99,
          estimatedDays: '5-7',
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        },
        {
          id: 2,
          name: 'Express Shipping',
          description: 'Fast delivery within 2-3 business days',
          cost: 12.99,
          estimatedDays: '2-3',
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        },
        {
          id: 3,
          name: 'Overnight Shipping',
          description: 'Next business day delivery',
          cost: 24.99,
          estimatedDays: '1',
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        },
        {
          id: 4,
          name: 'Free Shipping',
          description: 'Free delivery for orders over $50',
          cost: 0.00,
          estimatedDays: '7-10',
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        }
      ];

      defaultMethods.forEach(method => shippingCollection.insert(method));
      
      await new Promise<void>((resolve, reject) => {
        db.saveDatabase((err) => {
          if (err) reject(err);
          else resolve();
        });
      });
    }

    const shippingMethods = shippingCollection.find({});
    return NextResponse.json(shippingMethods);

  } catch (error) {
    console.error('Error fetching shipping methods:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

// POST - Create new shipping method
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { name, description, cost, estimatedDays, isActive } = body;

    // Validation
    if (!name || !description || cost === undefined || !estimatedDays) {
      return NextResponse.json(
        { error: 'Missing required fields: name, description, cost, estimatedDays' },
        { status: 400 }
      );
    }

    const db = await initializeDatabase();
    let shippingCollection = db.getCollection('shipping_methods');

    if (!shippingCollection) {
      shippingCollection = db.addCollection('shipping_methods');
    }

    // Check if shipping method with same name exists
    const existingMethod = shippingCollection.findOne({ name });
    if (existingMethod) {
      return NextResponse.json(
        { error: 'Shipping method with this name already exists' },
        { status: 409 }
      );
    }

    // Generate new ID
    const allMethods = shippingCollection.find({});
    const maxId = allMethods.length > 0 ? Math.max(...allMethods.map(m => m.id || 0)) : 0;
    const newId = maxId + 1;

    const newShippingMethod = {
      id: newId,
      name: name.trim(),
      description: description.trim(),
      cost: parseFloat(cost),
      estimatedDays: estimatedDays.trim(),
      isActive: isActive !== undefined ? isActive : true,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };

    shippingCollection.insert(newShippingMethod);

    // Save database
    await new Promise<void>((resolve, reject) => {
      db.saveDatabase((err) => {
        if (err) {
          console.error('Error saving database:', err);
          reject(err);
        } else {
          resolve();
        }
      });
    });

    return NextResponse.json({
      success: true,
      message: 'Shipping method created successfully',
      shippingMethod: newShippingMethod
    }, { status: 201 });

  } catch (error) {
    console.error('Error creating shipping method:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}