import { NextRequest, NextResponse } from 'next/server';
import { getDatabase } from '@/lib/database';

export async function POST(request: NextRequest) {
  console.log('Products API POST endpoint called');
  try {
    const db = await getDatabase();
    let products = db.getCollection('products');
    
    if (!products) {
      console.log('Products collection does not exist, creating...');
      products = db.addCollection('products');
    }
    
    const body = await request.json();
    console.log('Received product data:', body);
    
    // Get the type of product
    const { type, name, description, slug, categoryId, tags, isActive, isFeatured, metaTitle, metaDescription } = body;
    
    // Validate common required fields
    const missingFields = [];
    if (!name || name.trim() === '') missingFields.push('name');
    if (!description || description.trim() === '') missingFields.push('description');
    if (!categoryId && categoryId !== 0) missingFields.push('categoryId');
    
    // Type-specific validation
    if (type === 'simple') {
      const { price, sku, stockQuantity } = body;
      
      if (!price && price !== 0) missingFields.push('price');
      if (!sku || sku.trim() === '') missingFields.push('sku');
      if (stockQuantity === undefined || stockQuantity === null || stockQuantity === '') missingFields.push('stockQuantity');
      
      // Check if SKU already exists
      if (sku) {
        const existingProduct = products.findOne({ sku: sku });
        if (existingProduct) {
          return NextResponse.json(
            { error: 'A product with this SKU already exists' },
            { status: 400 }
          );
        }
      }
    } else if (type === 'variable') {
      const { attributes, variants } = body;
      
      if (!attributes || !Array.isArray(attributes) || attributes.length === 0) {
        missingFields.push('attributes');
      }
      
      if (!variants || !Array.isArray(variants) || variants.length === 0) {
        missingFields.push('variants');
      }
      
      // Check all variants have required fields
      if (variants && Array.isArray(variants)) {
        const invalidVariants = variants.filter(v => 
          !v.sku || !v.price || v.price <= 0 || v.stockQuantity === undefined
        );
        
        if (invalidVariants.length > 0) {
          missingFields.push('valid variant data (price, sku, stock quantity)');
        }
        
        // Check for duplicate variant SKUs
        const skus = variants.map(v => v.sku);
        const uniqueSkus = new Set(skus);
        if (skus.length !== uniqueSkus.size) {
          return NextResponse.json(
            { error: 'Each variant must have a unique SKU' },
            { status: 400 }
          );
        }
        
        // Check if any variant SKU already exists
        for (const variant of variants) {
          const existingProduct = products.findOne({ sku: variant.sku });
          if (existingProduct) {
            return NextResponse.json(
              { error: `A product with the SKU "${variant.sku}" already exists` },
              { status: 400 }
            );
          }
        }
      }
    } else {
      return NextResponse.json(
        { error: 'Invalid product type. Must be "simple" or "variable".' },
        { status: 400 }
      );
    }
    
    if (missingFields.length > 0) {
      console.log('Missing fields:', missingFields);
      return NextResponse.json(
        { error: `Missing required fields: ${missingFields.join(', ')}` },
        { status: 400 }
      );
    }

    // Check if slug already exists
    const existingSlug = products.findOne({ slug: slug });
    if (existingSlug) {
      return NextResponse.json(
        { error: 'A product with this URL slug already exists' },
        { status: 400 }
      );
    }

    // Verify category exists
    const categories = db.getCollection('categories');
    const category = categories.findOne({ '$loki': categoryId });
    if (!category) {
      return NextResponse.json(
        { error: 'Selected category does not exist' },
        { status: 400 }
      );
    }

    // Create new product based on type
    let newProduct;
    
    const commonFields = {
      id: Date.now(), // Simple ID generation
      name: name.trim(),
      description: description.trim(),
      slug: slug.trim(),
      categoryId: parseInt(categoryId),
      categoryName: category.name,
      isActive: Boolean(isActive),
      isFeatured: Boolean(isFeatured),
      tags: Array.isArray(tags) ? tags : [],
      metaTitle: metaTitle ? metaTitle.trim() : name.trim(),
      metaDescription: metaDescription ? metaDescription.trim() : description.trim().substring(0, 160),
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      type: type
    };
    
    if (type === 'simple') {
      const { price, comparePrice, sku, barcode, stockQuantity, lowStockAlert, weight, dimensions } = body;
      
      newProduct = {
        ...commonFields,
        price: parseFloat(price),
        comparePrice: comparePrice ? parseFloat(comparePrice) : null,
        sku: sku.trim().toUpperCase(),
        barcode: barcode ? barcode.trim() : null,
        stockQuantity: parseInt(stockQuantity),
        lowStockAlert: parseInt(lowStockAlert) || 5,
        weight: weight ? parseFloat(weight) : null,
        dimensions: dimensions ? dimensions.trim() : null,
      };
    } else { // variable product
      const { attributes, variants, priceRange, compareAtRange, lowStockAlert } = body;
      
      // Calculate price ranges if not provided
      let calculatedPriceRange = priceRange;
      let calculatedCompareAtRange = compareAtRange;
      
      if (!calculatedPriceRange) {
        const prices = variants.map((v: any) => v.price).filter((p: number) => p && p > 0);
        if (prices.length > 0) {
          calculatedPriceRange = {
            min: Math.min(...prices),
            max: Math.max(...prices)
          };
        }
      }
      
      if (!calculatedCompareAtRange) {
        const comparePrices = variants
          .map((v: any) => v.comparePrice)
          .filter((p: number | undefined): p is number => p !== undefined && p > 0);
        
        if (comparePrices.length > 0) {
          calculatedCompareAtRange = {
            min: Math.min(...comparePrices),
            max: Math.max(...comparePrices)
          };
        }
      }
      
      newProduct = {
        ...commonFields,
        attributes,
        variants,
        priceRange: calculatedPriceRange,
        compareAtRange: calculatedCompareAtRange,
        lowStockAlert: parseInt(lowStockAlert) || 5
      };
    }

    const insertedProduct = products.insert(newProduct);
    console.log('Product inserted:', insertedProduct);
    
    db.saveDatabase();
    console.log('Database saved successfully');

    return NextResponse.json(
      { 
        message: 'Product created successfully',
        product: insertedProduct
      },
      { status: 201 }
    );

  } catch (error) {
    console.error('Error creating product:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

export async function GET() {
  try {
    const db = await getDatabase();
    let products = db.getCollection('products');
    
    // If products collection doesn't exist, create it with sample data
    if (!products) {
      products = db.addCollection('products');
      
      // Add sample products
      const sampleProducts = [
        {
          id: 1,
          type: 'simple',
          name: 'Wireless Bluetooth Headphones',
          description: 'High-quality wireless headphones with noise cancellation and 30-hour battery life. Perfect for music lovers and professionals.',
          slug: 'wireless-bluetooth-headphones',
          price: 299.99,
          comparePrice: 399.99,
          categoryId: 1,
          categoryName: 'Electronics',
          sku: 'WBH-001',
          barcode: '1234567890123',
          stockQuantity: 45,
          lowStockAlert: 5,
          weight: 0.25,
          dimensions: '20 x 15 x 8 cm',
          isActive: true,
          isFeatured: true,
          tags: ['wireless', 'bluetooth', 'headphones', 'audio'],
          metaTitle: 'Premium Wireless Bluetooth Headphones - High Quality Audio',
          metaDescription: 'Discover our premium wireless Bluetooth headphones with superior sound quality, noise cancellation, and long battery life.',
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        },
        {
          id: 2,
          type: 'simple',
          name: 'Smart Fitness Watch',
          description: 'Advanced fitness tracking watch with heart rate monitoring, GPS, and smartphone notifications. Track your health 24/7.',
          slug: 'smart-fitness-watch',
          price: 199.99,
          comparePrice: 249.99,
          categoryId: 1,
          categoryName: 'Electronics',
          sku: 'SFW-002',
          barcode: '2345678901234',
          stockQuantity: 23,
          lowStockAlert: 5,
          weight: 0.05,
          dimensions: '4 x 4 x 1 cm',
          isActive: true,
          isFeatured: false,
          tags: ['smartwatch', 'fitness', 'health', 'wearable'],
          metaTitle: 'Smart Fitness Watch - Advanced Health Tracking',
          metaDescription: 'Stay healthy with our smart fitness watch featuring heart rate monitoring, GPS tracking, and comprehensive health insights.',
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        },
        {
          id: 3,
          type: 'simple',
          name: 'Professional Laptop Backpack',
          description: 'Durable and stylish laptop backpack with multiple compartments, USB charging port, and water-resistant material.',
          slug: 'professional-laptop-backpack',
          price: 79.99,
          comparePrice: null,
          categoryId: 2,
          categoryName: 'Fashion',
          sku: 'PLB-003',
          barcode: '3456789012345',
          stockQuantity: 67,
          lowStockAlert: 10,
          weight: 1.2,
          dimensions: '45 x 30 x 15 cm',
          isActive: true,
          isFeatured: false,
          tags: ['backpack', 'laptop', 'professional', 'travel'],
          metaTitle: 'Professional Laptop Backpack - Durable & Stylish',
          metaDescription: 'Organize your work essentials with our professional laptop backpack featuring multiple compartments and modern design.',
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        },
        {
          id: 4,
          type: 'variable',
          name: 'Premium T-Shirt',
          description: 'High-quality cotton t-shirt available in multiple colors and sizes. Perfect for casual wear or as a base layer.',
          slug: 'premium-t-shirt',
          categoryId: 2,
          categoryName: 'Fashion',
          attributes: [
            {
              id: 'attr_1',
              name: 'Size',
              values: ['S', 'M', 'L', 'XL']
            },
            {
              id: 'attr_2',
              name: 'Color',
              values: ['Red', 'Blue', 'Black', 'White']
            }
          ],
          variants: [
            {
              id: 'var_1',
              attributes: { 'attr_1': 'S', 'attr_2': 'Red' },
              price: 19.99,
              comparePrice: 24.99,
              stockQuantity: 15,
              sku: 'TS-S-RED',
              isActive: true
            },
            {
              id: 'var_2',
              attributes: { 'attr_1': 'M', 'attr_2': 'Red' },
              price: 19.99,
              comparePrice: 24.99,
              stockQuantity: 10,
              sku: 'TS-M-RED',
              isActive: true
            },
            {
              id: 'var_3',
              attributes: { 'attr_1': 'L', 'attr_2': 'Red' },
              price: 19.99,
              comparePrice: 24.99,
              stockQuantity: 5,
              sku: 'TS-L-RED',
              isActive: true
            },
            {
              id: 'var_4',
              attributes: { 'attr_1': 'S', 'attr_2': 'Blue' },
              price: 19.99,
              comparePrice: 24.99,
              stockQuantity: 8,
              sku: 'TS-S-BLUE',
              isActive: true
            },
            {
              id: 'var_5',
              attributes: { 'attr_1': 'M', 'attr_2': 'Blue' },
              price: 19.99,
              comparePrice: 24.99,
              stockQuantity: 12,
              sku: 'TS-M-BLUE',
              isActive: true
            }
          ],
          priceRange: { min: 19.99, max: 19.99 },
          compareAtRange: { min: 24.99, max: 24.99 },
          lowStockAlert: 3,
          isActive: true,
          isFeatured: true,
          tags: ['clothing', 't-shirt', 'cotton', 'casual'],
          metaTitle: 'Premium Cotton T-Shirt - Multiple Colors and Sizes',
          metaDescription: 'Shop our premium cotton t-shirts available in various colors and sizes. Perfect for everyday wear with superior comfort and durability.',
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        },
        {
          id: 5,
          type: 'simple',
          name: 'Portable Bluetooth Speaker',
          description: 'Compact waterproof Bluetooth speaker with 360-degree sound and 12-hour battery life. Perfect for outdoor adventures.',
          slug: 'portable-bluetooth-speaker',
          price: 149.99,
          comparePrice: 179.99,
          categoryId: 1,
          categoryName: 'Electronics',
          sku: 'PBS-004',
          barcode: '4567890123456',
          stockQuantity: 12,
          lowStockAlert: 15,
          weight: 0.6,
          dimensions: '12 x 12 x 6 cm',
          isActive: true,
          isFeatured: true,
          tags: ['speaker', 'bluetooth', 'portable', 'waterproof'],
          metaTitle: 'Portable Bluetooth Speaker - Waterproof & Powerful',
          metaDescription: 'Take your music anywhere with our portable waterproof Bluetooth speaker featuring 360-degree sound and long battery life.',
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        }
      ];
      
      products.insert(sampleProducts);
      db.saveDatabase();
    }

    const allProducts = products.find({});
    
    return NextResponse.json({
      products: allProducts,
      total: allProducts.length
    });

  } catch (error) {
    console.error('Error fetching products:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}