import { NextRequest, NextResponse } from 'next/server';
import { getDatabase } from '../../../lib/database';

export interface OrderItem {
  id: number;
  name: string;
  slug: string;
  price: number;
  quantity: number;
  comparePrice?: number;
}

export interface Order {
  $loki?: number;
  id?: number;
  orderNumber: string;
  customerInfo: {
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
  };
  shippingAddress: {
    address: string;
    city: string;
    state: string;
    zipCode: string;
    country: string;
  };
  items: OrderItem[];
  pricing: {
    subtotal: number;
    shipping: number;
    tax: number;
    codFee: number;
    total: number;
  };
  paymentMethod: 'stripe' | 'paypal' | 'card' | 'cod';
  paymentStatus: 'pending' | 'paid' | 'failed';
  orderStatus: 'pending' | 'processing' | 'shipped' | 'delivered' | 'cancelled';
  createdAt: string;
  updatedAt: string;
}

// Generate order number
function generateOrderNumber(): string {
  const year = new Date().getFullYear();
  const month = String(new Date().getMonth() + 1).padStart(2, '0');
  const day = String(new Date().getDate()).padStart(2, '0');
  const random = Math.floor(Math.random() * 10000).toString().padStart(4, '0');
  return `ORD-${year}${month}${day}-${random}`;
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const {
      customerInfo,
      shippingAddress,
      items,
      pricing,
      paymentMethod
    } = body;

    // Validation
    if (!customerInfo || !shippingAddress || !items || !pricing || !paymentMethod) {
      return NextResponse.json(
        { error: 'Missing required order information' },
        { status: 400 }
      );
    }

    if (!items.length) {
      return NextResponse.json(
        { error: 'Order must contain at least one item' },
        { status: 400 }
      );
    }

    const db = await getDatabase();
    let orders = db.getCollection<Order>('orders');

    if (!orders) {
      orders = db.addCollection<Order>('orders', { indices: ['orderNumber'] });
    }

    const now = new Date().toISOString();
    const orderNumber = generateOrderNumber();

    const newOrder: Order = {
      orderNumber,
      customerInfo,
      shippingAddress,
      items,
      pricing,
      paymentMethod,
      paymentStatus: paymentMethod === 'cod' ? 'pending' : 'pending',
      orderStatus: 'pending',
      createdAt: now,
      updatedAt: now,
    };

    const insertedOrder = orders.insert(newOrder);
    db.saveDatabase();

    return NextResponse.json({
      success: true,
      order: insertedOrder,
      message: 'Order created successfully'
    });

  } catch (error) {
    console.error('Error creating order:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const status = searchParams.get('status');
    const customerEmail = searchParams.get('customer');

    const db = await getDatabase();
    let orders = db.getCollection<Order>('orders');

    if (!orders) {
      return NextResponse.json([]);
    }

    let allOrders = orders.find({});

    // Filter by status if provided
    if (status && status !== 'all') {
      allOrders = allOrders.filter(order => order.orderStatus === status);
    }

    // Filter by customer email if provided
    if (customerEmail) {
      allOrders = allOrders.filter(order => 
        order.customerInfo.email.toLowerCase().includes(customerEmail.toLowerCase())
      );
    }

    // Sort by creation date (newest first)
    allOrders.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());

    return NextResponse.json(allOrders);

  } catch (error) {
    console.error('Error fetching orders:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}