import { NextRequest, NextResponse } from 'next/server';
import { getDatabase } from '../../../lib/database';
import { Order } from '../orders/route';

export interface Customer {
  id: string;
  firstName: string;
  lastName: string;
  email: string;
  phone: string | null;
  address: {
    street: string;
    city: string;
    state: string;
    zipCode: string;
    country: string;
  } | null;
  totalOrders: number;
  totalSpent: number;
  lastOrderDate: string | null;
  status: 'active' | 'inactive';
  createdAt: string;
  updatedAt: string;
}

export async function GET(request: NextRequest) {
  try {
    const db = await getDatabase();
    let orders = db.getCollection<Order>('orders');
    let customersCollection = db.getCollection<Customer>('customers');

    const customerMap = new Map<string, Customer>();

    // First, load customers from the customers collection
    if (customersCollection) {
      const allCustomers = customersCollection.find({});
      allCustomers.forEach(customer => {
        customerMap.set(customer.email.toLowerCase(), {
          ...customer,
          email: customer.email.toLowerCase()
        });
      });
    }

    // Then, update customer data with order information if orders exist
    if (orders) {
      const allOrders = orders.find({});
      
      // Group orders by customer email to update customer records

      allOrders.forEach(order => {
        const email = order.customerInfo.email.toLowerCase();
        
        if (customerMap.has(email)) {
          // Update existing customer with order data
          const customer = customerMap.get(email)!;
          customer.totalOrders += 1;
          customer.totalSpent += order.pricing.total;
          
          // Update last order date if this order is more recent
          const orderDate = new Date(order.createdAt);
          const lastOrderDate = customer.lastOrderDate ? new Date(customer.lastOrderDate) : null;
          
          if (!lastOrderDate || orderDate > lastOrderDate) {
            customer.lastOrderDate = order.createdAt;
          }
          
          // Update customer info with latest order data if needed
          if (!customer.address && order.shippingAddress) {
            customer.address = {
              street: order.shippingAddress.address,
              city: order.shippingAddress.city,
              state: order.shippingAddress.state,
              zipCode: order.shippingAddress.zipCode,
              country: order.shippingAddress.country
            };
          }
          customer.updatedAt = order.createdAt;
          
        } else {
          // Create new customer from order (if not in customers collection)
          const customer: Customer = {
            id: email, // Use email as unique ID
            firstName: order.customerInfo.firstName,
            lastName: order.customerInfo.lastName,
            email: order.customerInfo.email,
            phone: order.customerInfo.phone || null,
            address: {
              street: order.shippingAddress.address,
              city: order.shippingAddress.city,
              state: order.shippingAddress.state,
              zipCode: order.shippingAddress.zipCode,
              country: order.shippingAddress.country
            },
            totalOrders: 1,
            totalSpent: order.pricing.total,
            lastOrderDate: order.createdAt,
            status: 'active', // Default to active for customers with orders
            createdAt: order.createdAt,
            updatedAt: order.createdAt
          };
          
          customerMap.set(email, customer);
        }
      });
    }

    // Convert map to array and determine status based on recent activity
    const customersList = Array.from(customerMap.values()).map(customer => {
      // Mark as inactive if last order was more than 6 months ago
      const sixMonthsAgo = new Date();
      sixMonthsAgo.setMonth(sixMonthsAgo.getMonth() - 6);
      
      if (customer.lastOrderDate && new Date(customer.lastOrderDate) < sixMonthsAgo) {
        customer.status = 'inactive';
      }
      
      return customer;
    });

    // Sort by creation date (newest first)
    customersList.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());

    return NextResponse.json({
      success: true,
      customers: customersList
    });

  } catch (error) {
    console.error('Error fetching customers:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}