import { NextRequest, NextResponse } from 'next/server';
import { getDatabase } from '@/lib/database';

export interface Category {
  id?: number;
  name: string;
  description: string;
  icon: string;
  slug: string;
  isActive: boolean;
  sortOrder: number;
  createdAt: Date;
  updatedAt: Date;
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { name, description, icon, slug, isActive, sortOrder } = body;

    // Validate required fields
    if (!name || !description || !icon) {
      return NextResponse.json(
        { error: 'Name, description, and icon are required' },
        { status: 400 }
      );
    }

    // Validate name length
    if (name.length < 2 || name.length > 50) {
      return NextResponse.json(
        { error: 'Category name must be between 2 and 50 characters' },
        { status: 400 }
      );
    }

    // Validate description length
    if (description.length < 10 || description.length > 200) {
      return NextResponse.json(
        { error: 'Description must be between 10 and 200 characters' },
        { status: 400 }
      );
    }

    // Get database and categories collection
    const db = await getDatabase();
    let categories = db.getCollection<Category>('categories');
    
    if (!categories) {
      categories = db.addCollection<Category>('categories', { 
        indices: ['name', 'slug'] 
      });
    }

    // Check if category name already exists
    const existingByName = categories.findOne({ name: name.trim() });
    if (existingByName) {
      return NextResponse.json(
        { error: 'A category with this name already exists' },
        { status: 409 }
      );
    }

    // Check if slug already exists
    const existingBySlug = categories.findOne({ slug: slug.trim() });
    if (existingBySlug) {
      return NextResponse.json(
        { error: 'A category with this URL slug already exists' },
        { status: 409 }
      );
    }

    // Create new category
    const newCategory: Category = {
      name: name.trim(),
      description: description.trim(),
      icon: icon.trim(),
      slug: slug.trim(),
      isActive: Boolean(isActive),
      sortOrder: Number(sortOrder) || 0,
      createdAt: new Date(),
      updatedAt: new Date()
    };

    const insertedCategory = categories.insert(newCategory);
    db.saveDatabase();

    return NextResponse.json({
      message: 'Category created successfully',
      category: insertedCategory
    }, { status: 201 });

  } catch (error) {
    console.error('Category creation error:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const active = searchParams.get('active');
    
    // Get database and categories collection
    const db = await getDatabase();
    let categories = db.getCollection<Category>('categories');
    
    if (!categories) {
      categories = db.addCollection<Category>('categories', { 
        indices: ['name', 'slug'] 
      });
    }

    // Get all categories from the collection
    let allCategories = categories.find({});
    
    // If no categories exist, this means the database is empty
    if (allCategories.length === 0) {
      return NextResponse.json({
        categories: [],
        total: 0,
        message: 'No categories found. Please populate the database.'
      });
    }

    // Filter categories if active parameter is provided
    if (active === 'true') {
      allCategories = categories.find({ isActive: true });
    }

    // Sort by sortOrder (if exists), then by name
    allCategories.sort((a, b) => {
      const aSort = a.sortOrder || 0;
      const bSort = b.sortOrder || 0;
      if (aSort !== bSort) {
        return aSort - bSort;
      }
      return a.name.localeCompare(b.name);
    });

    // Add string ID for frontend consistency while keeping $loki
    const categoriesWithId = allCategories.map(category => ({
      ...category,
      id: category.id || category.$loki?.toString() || String(category.$loki)
    }));

    return NextResponse.json({
      categories: categoriesWithId,
      total: categoriesWithId.length
    });

  } catch (error) {
    console.error('Categories fetch error:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}