const fs = require('fs');
const path = require('path');

// Read the existing database file
const dbPath = path.join(__dirname, 'users.db');
let dbData;

try {
  const rawData = fs.readFileSync(dbPath, 'utf8');
  dbData = JSON.parse(rawData);
} catch (error) {
  console.error('Error reading database:', error);
  process.exit(1);
}

// Find or create collections
let categoriesCollection = dbData.collections.find(c => c.name === 'categories');
let subcategoriesCollection = dbData.collections.find(c => c.name === 'subcategories');

if (!categoriesCollection) {
  categoriesCollection = {
    name: 'categories',
    data: [],
    idIndex: 1,
    binaryIndices: {},
    constraints: {},
    uniqueNames: [],
    transforms: {},
    objType: 'categories',
    dirty: true,
    cachedIndex: null,
    cachedBinaryIndex: null,
    cachedData: null,
    adaptiveBinaryIndices: true,
    transactional: false,
    cloneObjects: false,
    cloneMethod: 'parse-stringify',
    asyncListeners: false,
    disableMeta: false,
    disableChangesApi: true,
    disableDeltaChangesApi: true,
    autoupdate: false,
    serializableIndices: true,
    disableFreeze: true,
    ttl: null,
    maxId: 0,
    DynamicViews: [],
    events: {},
    changes: [],
    dirtyIds: []
  };
  dbData.collections.push(categoriesCollection);
}

if (!subcategoriesCollection) {
  subcategoriesCollection = {
    name: 'subcategories',
    data: [],
    idIndex: 1,
    binaryIndices: {},
    constraints: {},
    uniqueNames: [],
    transforms: {},
    objType: 'subcategories',
    dirty: true,
    cachedIndex: null,
    cachedBinaryIndex: null,
    cachedData: null,
    adaptiveBinaryIndices: true,
    transactional: false,
    cloneObjects: false,
    cloneMethod: 'parse-stringify',
    asyncListeners: false,
    disableMeta: false,
    disableChangesApi: true,
    disableDeltaChangesApi: true,
    autoupdate: false,
    serializableIndices: true,
    disableFreeze: true,
    ttl: null,
    maxId: 0,
    DynamicViews: [],
    events: {},
    changes: [],
    dirtyIds: []
  };
  dbData.collections.push(subcategoriesCollection);
}

// 🔄 Reset collection (better than just clearing data)
function resetCollection(collection) {
  collection.data = [];
  collection.maxId = 0;
  collection.idIndex = 1;
  collection.cachedIndex = null;
  collection.cachedBinaryIndex = null;
  collection.cachedData = null;
  collection.dirty = true;
  collection.dirtyIds = [];
  collection.binaryIndices = {};
}

// ✅ Full reset to avoid duplicate or corrupted state
resetCollection(categoriesCollection);
resetCollection(subcategoriesCollection);

console.log('🏗️  Creating comprehensive categories and subcategories...');

// You still need to define this array:
const categoriesWithSubcategories = [/*... your category + subcategory objects here ...*/];

// Additional subcategories
const additionalSubcategories = {
  'Electronics': [
    { name: 'Drones', description: 'Quadcopters and drone accessories', icon: '🚁' },
    { name: 'VR & AR Devices', description: 'Virtual and augmented reality gadgets', icon: '🕶️' },
    { name: '3D Printers', description: '3D printers and printing materials', icon: '🖨️' }
  ],
  'Fashion & Clothing': [
    { name: 'Ethnic Wear', description: 'Traditional and cultural clothing', icon: '🎎' },
    { name: 'Activewear', description: 'Fitness and sports apparel', icon: '🧘' },
    { name: 'Costumes', description: 'Themed outfits and costumes', icon: '🥸' }
  ],
  'Home & Garden': [
    { name: 'Home Improvement', description: 'DIY and renovation supplies', icon: '🛠️' },
    { name: 'Smart Lighting', description: 'Automated lighting systems', icon: '💡' },
    { name: 'Cleaning Supplies', description: 'Cleaning tools and products', icon: '🧹' }
  ],
  'Sports & Outdoors': [
    { name: 'Hunting & Fishing', description: 'Gear for hunting and fishing', icon: '🎣' },
    { name: 'Climbing & Mountaineering', description: 'Ropes, harnesses, and climbing gear', icon: '🧗' },
    { name: 'Skateboarding & Scooters', description: 'Boards, scooters, and accessories', icon: '🛹' }
  ],
  'Books & Media': [
    { name: 'Reference Books', description: 'Dictionaries, encyclopedias, and manuals', icon: '📙' },
    { name: 'Language Learning', description: 'Books and tools for learning new languages', icon: '🈷️' },
    { name: 'Sheet Music', description: 'Musical notation and learning resources', icon: '🎼' }
  ],
  'Health & Beauty': [
    { name: 'Spa & Relaxation', description: 'Spa kits and relaxation products', icon: '🛁' },
    { name: 'Cosmetic Tools', description: 'Brushes, applicators, and tools', icon: '🪞' },
    { name: 'Feminine Care', description: 'Hygiene and care products for women', icon: '🩲' }
  ],
  'Automotive': [
    { name: 'Car Lighting', description: 'Headlights, taillights, and bulbs', icon: '💡' },
    { name: 'Performance Parts', description: 'Aftermarket and upgrade parts', icon: '🏎️' },
    { name: 'Dash Cameras', description: 'Cameras and safety recorders', icon: '🎥' }
  ],
  'Food & Beverages': [
    { name: 'Gourmet & Specialty', description: 'Delicacies and fine foods', icon: '🧀' },
    { name: 'Meal Kits', description: 'Prepped meal boxes and recipes', icon: '🍱' },
    { name: 'Beverage Accessories', description: 'Coffee makers, tea infusers, and more', icon: '🫖' }
  ],
  'Toys & Games': [
    { name: 'STEM Toys', description: 'Science, tech, engineering & math toys', icon: '🔧' },
    { name: 'Musical Toys', description: 'Instruments and sound-making toys', icon: '🎼' },
    { name: 'Fidget Toys', description: 'Sensory and fidget tools', icon: '🌀' }
  ],
  'Pet Supplies': [
    { name: 'Pet Furniture', description: 'Beds, houses, and play areas for pets', icon: '🛏️' },
    { name: 'Pet Apparel', description: 'Clothing and accessories for pets', icon: '👚' },
    { name: 'Pet Tech', description: 'Smart feeders, GPS trackers, and gadgets', icon: '📡' }
  ],
  'Office & School': [
    { name: 'Whiteboards & Boards', description: 'Boards for writing and planning', icon: '🧾' },
    { name: 'School Uniforms', description: 'Uniforms and schoolwear', icon: '👕' },
    { name: 'Exam Supplies', description: 'Tools for tests and exams', icon: '📝' }
  ],
  'Travel & Luggage': [
    { name: 'Travel Insurance', description: 'Coverage for trips and emergencies', icon: '🛡️' },
    { name: 'Luggage Tags & Locks', description: 'Identification and security', icon: '🔐' },
    { name: 'Packing Organizers', description: 'Bags and cubes for organizing', icon: '🧦' }
  ]
};

// Merge additional subcategories
categoriesWithSubcategories.forEach(category => {
  const additions = additionalSubcategories[category.name];
  if (additions && Array.isArray(additions)) {
    category.subcategories.push(...additions);
  }
});

let categoryCount = 0;
let subcategoryCount = 0;

// Insert categories and subcategories
categoriesWithSubcategories.forEach((catData, catIndex) => {
  const categoryId = `CAT-${Date.now()}-${(categoryCount + 1).toString().padStart(3, '0')}`;
  const category = {
    id: categoryId,
    name: catData.name,
    description: catData.description,
    slug: catData.name.toLowerCase().replace(/[^a-z0-9 -]/g, '').replace(/\s+/g, '-').replace(/-+/g, '-').trim(),
    icon: catData.icon,
    isActive: true,
    sortOrder: catIndex + 1,
    createdAt: new Date(Date.now() - Math.random() * 90 * 24 * 60 * 60 * 1000).toISOString(),
    updatedAt: new Date().toISOString(),
    meta: {
      revision: 0,
      created: Date.now(),
      version: 0
    },
    $loki: categoryCount + 1
  };

  categoriesCollection.data.push(category);
  categoryCount++;

  catData.subcategories.forEach((subcatData, subcatIndex) => {
    const subcategory = {
      id: `SUB-${Date.now()}-${(subcategoryCount + 1).toString().padStart(4, '0')}`,
      categoryId: category.id,
      name: subcatData.name,
      description: subcatData.description,
      slug: subcatData.name.toLowerCase().replace(/[^a-z0-9 -]/g, '').replace(/\s+/g, '-').replace(/-+/g, '-').trim(),
      icon: subcatData.icon,
      isActive: true,
      sortOrder: subcatIndex + 1,
      createdAt: new Date(Date.now() - Math.random() * 90 * 24 * 60 * 60 * 1000).toISOString(),
      updatedAt: new Date().toISOString(),
      meta: {
        revision: 0,
        created: Date.now(),
        version: 0
      },
      $loki: subcategoryCount + 1
    };

    subcategoriesCollection.data.push(subcategory);
    subcategoryCount++;
  });
});

try {
  fs.writeFileSync(dbPath, JSON.stringify(dbData, null, 2));
  console.log(`✅ Successfully created ${categoryCount} categories and ${subcategoryCount} subcategories.`);
} catch (error) {
  console.error('Error writing updated database:', error);
}
