const loki = require('lokijs');

// Initialize database
const db = new loki('users.db');

db.loadDatabase({}, () => {
  console.log('🚀 GENERATING 1000 DUMMY PRODUCTS');
  console.log('==================================\n');
  
  const categories = db.getCollection('categories');
  const subcategories = db.getCollection('subcategories');
  let products = db.getCollection('products');
  
  if (!categories || !subcategories) {
    console.error('❌ Categories or subcategories collection not found');
    return;
  }
  
  if (!products) {
    products = db.addCollection('products');
  }
  
  const allCategories = categories.find({});
  const allSubcategories = subcategories.find({});
  
  console.log(`📊 Found ${allCategories.length} categories and ${allSubcategories.length} subcategories`);
  
  // Clear existing products and keep track of original products
  const existingProducts = products.find({});
  console.log(`🗑️  Found ${existingProducts.length} existing products`);
  
  // Clear all products - we'll regenerate everything
  products.clear();
  console.log(`🗑️  Cleared all products\n`);
  
  // Product name templates for different categories
  const productTemplates = {
    // Electronics
    'audio-headphones': ['Wireless Headphones', 'Bluetooth Earbuds', 'Gaming Headset', 'Studio Monitors', 'Noise-Canceling Headphones'],
    'cameras': ['Digital Camera', 'DSLR Camera', 'Action Camera', 'Instant Camera', 'Security Camera'],
    'smart-watches': ['Fitness Tracker', 'Smart Watch', 'Sports Watch', 'Luxury Smart Watch', 'Kids Smart Watch'],
    'tv-home-theater': ['Smart TV', 'Sound Bar', 'Home Theater System', 'Streaming Device', 'TV Stand'],
    'computer-accessories': ['Wireless Mouse', 'Mechanical Keyboard', '4K Monitor', 'USB Hub', 'Laptop Stand'],
    'gaming-consoles': ['Gaming Console', 'Gaming Controller', 'VR Headset', 'Gaming Chair', 'Gaming Keyboard'],
    'cables-chargers': ['USB Cable', 'Wireless Charger', 'Power Bank', 'Car Charger', 'Lightning Cable'],
    'smartphones': ['Smartphone', 'Phone Case', 'Screen Protector', 'Phone Holder', 'Selfie Stick'],
    
    // Fashion
    'shoes': ['Running Shoes', 'Dress Shoes', 'Sneakers', 'Boots', 'Sandals'],
    'fashion-accessories': ['Handbag', 'Wallet', 'Belt', 'Scarf', 'Sunglasses'],
    'formal-wear': ['Business Suit', 'Dress Shirt', 'Evening Dress', 'Tie', 'Blazer'],
    'swimwear': ['Bikini', 'One-piece Swimsuit', 'Board Shorts', 'Rash Guard', 'Beach Cover-up'],
    'lingerie-underwear': ['Bra Set', 'Boxer Briefs', 'Thermal Underwear', 'Sports Bra', 'Silk Pajamas'],
    'activewear': ['Yoga Pants', 'Athletic T-shirt', 'Compression Shorts', 'Sports Jacket', 'Training Shoes'],
    
    // Home & Garden
    'home-furniture': ['Sofa', 'Dining Table', 'Office Chair', 'Bookshelf', 'Coffee Table'],
    'bedding-bath': ['Bed Sheets', 'Towel Set', 'Pillow', 'Shower Curtain', 'Bath Mat'],
    'storage-organization': ['Storage Box', 'Closet Organizer', 'Drawer Divider', 'Shelf Unit', 'Laundry Basket'],
    'tools-hardware': ['Drill Set', 'Hammer', 'Screwdriver Set', 'Measuring Tape', 'Tool Box'],
    'cleaning-supplies': ['Vacuum Cleaner', 'Cleaning Spray', 'Microfiber Cloth', 'Mop', 'Trash Can'],
    'lighting': ['LED Bulb', 'Table Lamp', 'Ceiling Light', 'Floor Lamp', 'Night Light'],
    
    // Sports
    'athletic-wear': ['Athletic Shorts', 'Running Shirt', 'Sports Bra', 'Training Pants', 'Workout Tank'],
    'running-cardio': ['Running Shoes', 'Treadmill', 'Heart Rate Monitor', 'Running Belt', 'Water Bottle'],
    'strength-training': ['Dumbbells', 'Resistance Bands', 'Weight Bench', 'Kettlebell', 'Pull-up Bar'],
    'yoga-pilates': ['Yoga Mat', 'Yoga Block', 'Pilates Ring', 'Yoga Strap', 'Meditation Cushion'],
    'golf': ['Golf Clubs', 'Golf Balls', 'Golf Bag', 'Golf Gloves', 'Golf Tees'],
    'tennis-racquet-sports': ['Tennis Racket', 'Tennis Balls', 'Badminton Set', 'Table Tennis Paddle', 'Squash Racket'],
    
    // Default templates for other categories
    'default': ['Premium Product', 'Deluxe Item', 'Professional Grade', 'Essential Tool', 'Quality Item']
  };
  
  // Brand names for variety
  const brands = [
    'ProMax', 'TechCore', 'UltraFit', 'SwiftPro', 'PrimeLine', 'EliteCore', 'MaxForce', 'PowerTech',
    'FlexiPro', 'SmartEdge', 'RapidMax', 'DurablePro', 'PrecisionPro', 'AdvancedTech', 'QualityMax',
    'InnovatePro', 'SuperiorMax', 'PremiumCore', 'ExpertLine', 'MasterCraft', 'ProGrade', 'TopTier'
  ];
  
  // Adjectives for product names
  const adjectives = [
    'Premium', 'Professional', 'Advanced', 'Ultra', 'Super', 'Mega', 'Deluxe', 'Elite',
    'Smart', 'Pro', 'Max', 'Plus', 'Enhanced', 'Superior', 'High-Performance', 'Compact',
    'Wireless', 'Portable', 'Durable', 'Lightweight', 'Heavy-Duty', 'Ergonomic'
  ];
  
  // Function to create slug from name
  function createSlug(name) {
    return name.toLowerCase()
      .replace(/[^a-z0-9 -]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim('-');
  }
  
  // Function to generate random price
  function generatePrice(basePrice = 50) {
    const multiplier = Math.random() * 10 + 0.5; // 0.5 to 10.5
    return Math.round(basePrice * multiplier * 100) / 100;
  }
  
  // Function to generate product description
  function generateDescription(productName, categoryName, subcategoryName) {
    const descriptions = [
      `High-quality ${productName.toLowerCase()} perfect for ${subcategoryName.toLowerCase()}. Features premium materials and excellent craftsmanship.`,
      `Professional-grade ${productName.toLowerCase()} designed for ${categoryName.toLowerCase()} enthusiasts. Durable and reliable performance.`,
      `Premium ${productName.toLowerCase()} with advanced features. Ideal for both beginners and professionals in ${subcategoryName.toLowerCase()}.`,
      `Top-rated ${productName.toLowerCase()} offering exceptional value. Perfect addition to your ${categoryName.toLowerCase()} collection.`,
      `Innovative ${productName.toLowerCase()} with cutting-edge technology. Designed for maximum performance and durability.`
    ];
    return descriptions[Math.floor(Math.random() * descriptions.length)];
  }
  
  let generatedProducts = 0;
  let nextProductId = 1;
  
  // Calculate products per subcategory (aim for ~4 products per subcategory)
  const productsPerSubcategory = Math.ceil(1000 / allSubcategories.length);
  
  console.log(`🎯 Generating ~${productsPerSubcategory} products per subcategory\n`);
  
  // Generate products for each subcategory
  allSubcategories.forEach((subcategory, subcategoryIndex) => {
    const category = allCategories.find(cat => cat.id === subcategory.categoryId);
    if (!category) return;
    
    const templates = productTemplates[subcategory.slug] || productTemplates['default'];
    
    for (let i = 0; i < productsPerSubcategory && generatedProducts < 1000; i++) {
      try {
        const template = templates[Math.floor(Math.random() * templates.length)];
        const brand = brands[Math.floor(Math.random() * brands.length)];
        const adjective = adjectives[Math.floor(Math.random() * adjectives.length)];
        
        const productName = `${brand} ${adjective} ${template}`;
        const basePrice = generatePrice();
        const comparePrice = Math.random() > 0.7 ? Math.round(basePrice * 1.3 * 100) / 100 : null;
        
        const product = {
          id: nextProductId++,
          name: productName,
          description: generateDescription(productName, category.name, subcategory.name),
          slug: createSlug(productName) + '-' + Math.random().toString(36).substr(2, 4),
          price: basePrice,
          comparePrice: comparePrice,
          categoryId: category.id,
          subcategoryId: subcategory.id,
          categoryName: category.name,
          subcategoryName: subcategory.name,
          sku: `${category.name.substr(0, 3).toUpperCase()}-${String(nextProductId).padStart(4, '0')}`,
          barcode: `${Math.floor(Math.random() * 9000000000000) + 1000000000000}`,
          stockQuantity: Math.floor(Math.random() * 100) + 10,
          lowStockAlert: 5,
          weight: Math.round((Math.random() * 5 + 0.1) * 100) / 100,
          dimensions: `${Math.floor(Math.random() * 30 + 5)} x ${Math.floor(Math.random() * 30 + 5)} x ${Math.floor(Math.random() * 20 + 2)} cm`,
          isActive: true,
          isFeatured: Math.random() > 0.8, // 20% chance of being featured
          tags: [
            subcategory.name.toLowerCase().replace(/\s+/g, '-'),
            category.name.toLowerCase().replace(/\s+/g, '-'),
            template.toLowerCase().replace(/\s+/g, '-')
          ],
          imageUrl: `/images/products/${createSlug(template)}.jpg`,
          metaTitle: `${productName} - ${category.name}`,
          metaDescription: `Shop ${productName} in our ${subcategory.name} collection. High quality ${category.name.toLowerCase()} products.`,
          createdAt: new Date(Date.now() - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString(), // Random date within last 30 days
          updatedAt: new Date().toISOString()
        };
        
        products.insert(product);
        generatedProducts++;
        
        if (generatedProducts % 100 === 0) {
          console.log(`✅ Generated ${generatedProducts} products...`);
        }
        
      } catch (error) {
        console.error(`❌ Error generating product for ${subcategory.name}:`, error);
      }
    }
  });
  
  // Save the database
  db.saveDatabase((err) => {
    if (err) {
      console.error('❌ Error saving database:', err);
    } else {
      console.log(`\n🎉 Successfully generated ${generatedProducts} new products!`);
      
      // Final verification
      const finalProductCount = products.count();
      const categoriesWithProducts = new Set();
      const subcategoriesWithProducts = new Set();
      
      products.find({}).forEach(product => {
        if (product.categoryId) categoriesWithProducts.add(product.categoryId);
        if (product.subcategoryId) subcategoriesWithProducts.add(product.subcategoryId);
      });
      
      console.log('\n📊 FINAL STATISTICS:');
      console.log('====================');
      console.log(`Total products: ${finalProductCount}`);
      console.log(`Categories with products: ${categoriesWithProducts.size}/${allCategories.length}`);
      console.log(`Subcategories with products: ${subcategoriesWithProducts.size}/${allSubcategories.length}`);
      console.log(`Featured products: ${products.find({ isFeatured: true }).length}`);
      console.log(`Active products: ${products.find({ isActive: true }).length}`);
      
      if (subcategoriesWithProducts.size === allSubcategories.length) {
        console.log('\n✅ SUCCESS: All subcategories now have products!');
      } else {
        console.log(`\n⚠️  ${allSubcategories.length - subcategoriesWithProducts.size} subcategories still need products`);
      }
      
      console.log('\n🛍️ PRODUCT GENERATION COMPLETE! Ready for shopping! 🚀');
    }
  });
});