const loki = require('lokijs');

const db = new loki('users.db');

db.loadDatabase({}, function() {
  console.log('🔧 Fixing category-subcategory relationships...');
  
  const categories = db.getCollection('categories');
  const subcategories = db.getCollection('subcategories');
  
  // Update categories to have proper string IDs that match their $loki values
  const allCategories = categories.find({});
  console.log(`\n📝 Updating ${allCategories.length} categories with proper IDs...`);
  
  allCategories.forEach(category => {
    // Set id to string version of $loki
    category.id = String(category.$loki);
    categories.update(category);
    console.log(`✅ Updated category: ${category.name} (id: ${category.id})`);
  });
  
  // Now check which subcategories need to be matched to existing categories
  const allSubcategories = subcategories.find({});
  console.log(`\n🔍 Checking ${allSubcategories.length} subcategories...`);
  
  let fixedCount = 0;
  let orphanedCount = 0;
  
  allSubcategories.forEach(subcategory => {
    // Find category by $loki matching the subcategory's categoryId
    const parentCategory = categories.findOne({$loki: subcategory.categoryId});
    
    if (parentCategory) {
      // Update subcategory to use string categoryId
      subcategory.categoryId = String(parentCategory.$loki);
      subcategories.update(subcategory);
      fixedCount++;
    } else {
      console.log(`⚠️  Orphaned subcategory: ${subcategory.name} (categoryId: ${subcategory.categoryId})`);
      orphanedCount++;
    }
  });
  
  console.log(`\n📊 Results:`);
  console.log(`✅ Fixed relationships: ${fixedCount} subcategories`);
  console.log(`⚠️  Orphaned subcategories: ${orphanedCount}`);
  
  // Save the database
  db.save(function() {
    console.log('💾 Database saved successfully!');
    
    // Verify the fixes
    console.log('\n🔍 Verification:');
    const verifyCategories = categories.find({});
    verifyCategories.slice(0, 5).forEach(cat => {
      const subCount = subcategories.find({categoryId: cat.id}).length;
      console.log(`✅ ${cat.icon} ${cat.name}: ${subCount} subcategories`);
    });
    
    process.exit(0);
  });
});