const loki = require('lokijs');

// Initialize database
const db = new loki('users.db');

db.loadDatabase({}, function() {
    console.log('🔧 Fixing category filter data type issue...\n');
    
    // Get collections
    const categoriesCollection = db.getCollection('categories');
    const subcategoriesCollection = db.getCollection('subcategories');
    
    if (!categoriesCollection || !subcategoriesCollection) {
        console.log('❌ Collections not found');
        return;
    }
    
    console.log('📊 Current state:');
    console.log(`  Categories: ${categoriesCollection.count()}`);
    console.log(`  Subcategories: ${subcategoriesCollection.count()}\n`);
    
    // Check data types
    const categories = categoriesCollection.find({});
    const subcategories = subcategoriesCollection.find({});
    
    console.log('🔍 Data type analysis:');
    if (categories.length > 0) {
        const sampleCategory = categories[0];
        console.log(`  Category ID type: ${typeof sampleCategory.$loki} (value: ${sampleCategory.$loki})`);
    }
    
    if (subcategories.length > 0) {
        const sampleSubcategory = subcategories[0];
        console.log(`  Subcategory categoryId type: ${typeof sampleSubcategory.categoryId} (value: ${sampleSubcategory.categoryId})`);
    }
    
    // Check for mismatches
    let fixedCount = 0;
    let issuesFound = [];
    
    subcategories.forEach(subcategory => {
        const categoryIdAsNumber = parseInt(subcategory.categoryId);
        const categoryIdAsString = subcategory.categoryId.toString();
        
        // Find the parent category
        const parentCategory = categoriesCollection.findOne({ $loki: categoryIdAsNumber });
        
        if (!parentCategory) {
            issuesFound.push(`Subcategory "${subcategory.name}" has categoryId "${subcategory.categoryId}" but no matching category found`);
        } else {
            // Check if categoryId needs to be converted to string for consistency
            if (typeof subcategory.categoryId !== 'string') {
                subcategory.categoryId = categoryIdAsString;
                subcategoriesCollection.update(subcategory);
                fixedCount++;
            }
        }
    });
    
    console.log('\n🔧 Issues and fixes:');
    if (issuesFound.length > 0) {
        console.log('  Issues found:');
        issuesFound.forEach(issue => console.log(`    ❌ ${issue}`));
    } else {
        console.log('  ✅ No orphaned subcategories found');
    }
    
    if (fixedCount > 0) {
        console.log(`  🔧 Fixed ${fixedCount} categoryId data types`);
        
        // Save the database
        db.saveDatabase(function(err) {
            if (err) {
                console.log('❌ Error saving database:', err);
            } else {
                console.log('  ✅ Database saved successfully');
                
                // Test the filtering logic
                console.log('\n🧪 Testing filter logic:');
                testFilterLogic();
            }
        });
    } else {
        console.log('  ✅ No data type fixes needed');
        // Test the filtering logic
        console.log('\n🧪 Testing filter logic:');
        testFilterLogic();
    }
    
    function testFilterLogic() {
        // Test category filtering like the frontend does
        const testCategoryId = categories[0].$loki.toString(); // Convert to string like frontend
        console.log(`  Testing filter with categoryId: "${testCategoryId}" (type: ${typeof testCategoryId})`);
        
        // This is how the frontend filters
        const filteredSubcategories = subcategories.filter(subcategory => 
            subcategory.categoryId === testCategoryId
        );
        
        console.log(`  Found ${filteredSubcategories.length} subcategories for category "${categories[0].name}"`);
        
        if (filteredSubcategories.length > 0) {
            console.log('  ✅ Category filtering is working!');
            filteredSubcategories.slice(0, 3).forEach(sub => {
                console.log(`    - ${sub.name} (categoryId: ${sub.categoryId})`);
            });
        } else {
            console.log('  ❌ Category filtering is not working - no subcategories found');
            
            // Show what we have for debugging
            console.log('  Debug info:');
            console.log(`    Looking for categoryId === "${testCategoryId}"`);
            console.log('    Available categoryIds in subcategories:');
            const uniqueCategoryIds = [...new Set(subcategories.map(sub => sub.categoryId))];
            uniqueCategoryIds.forEach(id => {
                console.log(`      "${id}" (type: ${typeof id})`);
            });
        }
    }
});