const Loki = require('lokijs');

console.log("🔧 Fixing category data structure and dates...");

const db = new Loki("users.db", {
  autoload: true,
  autoloadCallback: function () {
    const categories = db.getCollection("categories");
    
    if (!categories) {
      console.log("❌ Categories collection not found.");
      process.exit(1);
    }

    console.log(`Found ${categories.data.length} categories to check and fix`);

    let updatedCount = 0;

    categories.data.forEach((category) => {
      let needsUpdate = false;

      // Fix missing or invalid createdAt date
      if (!category.createdAt || category.createdAt === "Invalid Date" || isNaN(new Date(category.createdAt))) {
        category.createdAt = new Date().toISOString();
        needsUpdate = true;
      }

      // Fix missing or invalid updatedAt date
      if (!category.updatedAt || category.updatedAt === "Invalid Date" || isNaN(new Date(category.updatedAt))) {
        category.updatedAt = new Date().toISOString();
        needsUpdate = true;
      }

      // Fix missing slug
      if (!category.slug || category.slug === "") {
        category.slug = category.name.toLowerCase()
          .replace(/[^a-z0-9 -]/g, '')
          .replace(/\s+/g, '-')
          .replace(/-+/g, '-')
          .trim();
        needsUpdate = true;
      }

      // Fix missing description
      if (!category.description || category.description === "") {
        category.description = `${category.name} products and services`;
        needsUpdate = true;
      }

      // Fix missing icon
      if (!category.icon || category.icon === "undefined") {
        // Set appropriate icons for known categories
        const iconMap = {
          'Electronics': '📱',
          'Fashion': '👕',
          'Home & Kitchen': '🏠',
          'Documents / Files': '📁',
          'Shopping / Retail': '🛍️',
          'Clothing': '👕',
          'Footwear': '👟',
          'Mobile Devices': '📱',
          'Computers / Tech': '💻',
          'Gaming': '🎮',
          'Books / Education': '📚',
          'Music': '🎵',
          'Food': '🍔',
          'Coffee / Beverages': '☕'
        };
        category.icon = iconMap[category.name] || '📁';
        needsUpdate = true;
      }

      // Ensure isActive is properly set
      if (category.isActive === undefined || category.isActive === null) {
        category.isActive = true;
        needsUpdate = true;
      }

      // Ensure sortOrder is set
      if (category.sortOrder === undefined || category.sortOrder === null) {
        category.sortOrder = 0;
        needsUpdate = true;
      }

      if (needsUpdate) {
        categories.update(category);
        updatedCount++;
        console.log(`✅ Updated: ${category.name}`);
      }
    });

    db.saveDatabase(() => {
      console.log(`\n✅ Database update complete!`);
      console.log(`📊 Updated ${updatedCount} categories`);
      console.log(`📊 Total categories: ${categories.data.length}`);
      
      // Show sample of fixed data
      console.log("\n📋 Sample category after fixes:");
      const sampleCategory = categories.data[0];
      console.log(`• Name: ${sampleCategory.name}`);
      console.log(`• Slug: ${sampleCategory.slug}`);
      console.log(`• Icon: ${sampleCategory.icon}`);
      console.log(`• Description: ${sampleCategory.description}`);
      console.log(`• Created: ${sampleCategory.createdAt}`);
      console.log(`• Active: ${sampleCategory.isActive}`);
      console.log(`• Sort Order: ${sampleCategory.sortOrder}`);
      
      process.exit(0);
    });
  },
});