const loki = require('lokijs');

console.log('🎉 FINAL COMPLETE SYSTEM VERIFICATION');
console.log('=====================================\n');

const db = new loki('users.db');
db.loadDatabase({}, () => {
  const categories = db.getCollection('categories');
  const subcategories = db.getCollection('subcategories');
  const products = db.getCollection('products');
  
  if (!categories || !subcategories || !products) {
    console.error('❌ Collections not found');
    return;
  }
  
  const allCategories = categories.find({});
  const allSubcategories = subcategories.find({});
  const allProducts = products.find({});
  
  console.log('📊 SYSTEM OVERVIEW:');
  console.log('===================');
  console.log(`✅ Categories: ${allCategories.length} (Target: 35)`);
  console.log(`✅ Subcategories: ${allSubcategories.length} (Target: 244+)`);
  console.log(`✅ Products: ${allProducts.length} (Target: 1000+)`);
  
  // Check active items
  const activeCategories = allCategories.filter(c => c.isActive).length;
  const activeSubcategories = allSubcategories.filter(s => s.isActive).length;
  const activeProducts = allProducts.filter(p => p.isActive).length;
  const featuredProducts = allProducts.filter(p => p.isFeatured).length;
  
  console.log(`✅ Active Categories: ${activeCategories}`);
  console.log(`✅ Active Subcategories: ${activeSubcategories}`);
  console.log(`✅ Active Products: ${activeProducts}`);
  console.log(`✅ Featured Products: ${featuredProducts}`);
  
  // Check relationships
  const categoriesWithSubcategories = new Set();
  const subcategoriesWithProducts = new Set();
  const categoriesWithProducts = new Set();
  
  allSubcategories.forEach(sub => {
    if (sub.categoryId) categoriesWithSubcategories.add(sub.categoryId);
  });
  
  allProducts.forEach(product => {
    if (product.categoryId) categoriesWithProducts.add(product.categoryId);
    if (product.subcategoryId) subcategoriesWithProducts.add(product.subcategoryId);
  });
  
  console.log('\n🔗 RELATIONSHIP VERIFICATION:');
  console.log('=============================');
  console.log(`✅ Categories with subcategories: ${categoriesWithSubcategories.size}/${allCategories.length}`);
  console.log(`✅ Categories with products: ${categoriesWithProducts.size}/${allCategories.length}`);
  console.log(`✅ Subcategories with products: ${subcategoriesWithProducts.size}/${allSubcategories.length}`);
  
  // Check for orphaned data
  const orphanedSubcategories = allSubcategories.filter(sub => 
    !allCategories.find(cat => cat.id === sub.categoryId)
  );
  
  const orphanedProducts = allProducts.filter(product => 
    (!product.subcategoryId || !allSubcategories.find(sub => sub.id === product.subcategoryId)) ||
    (!product.categoryId || !allCategories.find(cat => cat.id === product.categoryId))
  );
  
  if (orphanedSubcategories.length === 0 && orphanedProducts.length === 0) {
    console.log('✅ No orphaned data found');
  } else {
    console.log(`⚠️  Orphaned subcategories: ${orphanedSubcategories.length}`);
    console.log(`⚠️  Orphaned products: ${orphanedProducts.length}`);
  }
  
  // Sample data verification
  console.log('\n📋 SAMPLE DATA:');
  console.log('===============');
  
  if (allCategories.length > 0) {
    const samples = allCategories.slice(0, 5);
    console.log('Sample Categories:');
    samples.forEach(cat => {
      const subCount = allSubcategories.filter(sub => sub.categoryId === cat.id).length;
      const prodCount = allProducts.filter(prod => prod.categoryId === cat.id).length;
      console.log(`  - ${cat.name} (${subCount} subcategories, ${prodCount} products)`);
    });
  }
  
  if (allSubcategories.length > 0) {
    console.log('\nSample Subcategories:');
    allSubcategories.slice(0, 5).forEach(sub => {
      const category = allCategories.find(cat => cat.id === sub.categoryId);
      const prodCount = allProducts.filter(prod => prod.subcategoryId === sub.id).length;
      console.log(`  - ${sub.name} (${category ? category.name : 'Unknown'}, ${prodCount} products)`);
    });
  }
  
  if (allProducts.length > 0) {
    console.log('\nSample Products:');
    allProducts.slice(0, 5).forEach(prod => {
      const category = allCategories.find(cat => cat.id === prod.categoryId);
      const subcategory = allSubcategories.find(sub => sub.id === prod.subcategoryId);
      console.log(`  - ${prod.name} ($${prod.price}) - ${category ? category.name : 'Unknown'} > ${subcategory ? subcategory.name : 'Unknown'}`);
    });
  }
  
  // Price analysis
  console.log('\n💰 PRICING ANALYSIS:');
  console.log('====================');
  
  const prices = allProducts.map(p => p.price).filter(p => p > 0);
  if (prices.length > 0) {
    const avgPrice = prices.reduce((sum, price) => sum + price, 0) / prices.length;
    const minPrice = Math.min(...prices);
    const maxPrice = Math.max(...prices);
    const productsWithComparePrice = allProducts.filter(p => p.comparePrice).length;
    
    console.log(`Average price: $${avgPrice.toFixed(2)}`);
    console.log(`Price range: $${minPrice.toFixed(2)} - $${maxPrice.toFixed(2)}`);
    console.log(`Products with compare prices: ${productsWithComparePrice} (${((productsWithComparePrice/allProducts.length)*100).toFixed(1)}%)`);
  }
  
  // Inventory analysis
  console.log('\n📦 INVENTORY STATUS:');
  console.log('===================');
  
  const totalStock = allProducts.reduce((sum, p) => sum + (p.stockQuantity || 0), 0);
  const lowStockProducts = allProducts.filter(p => p.stockQuantity <= 10).length;
  const outOfStockProducts = allProducts.filter(p => p.stockQuantity === 0).length;
  
  console.log(`Total inventory: ${totalStock.toLocaleString()} units`);
  console.log(`Low stock products (≤10): ${lowStockProducts}`);
  console.log(`Out of stock products: ${outOfStockProducts}`);
  
  // Final assessment
  console.log('\n🏆 SYSTEM ASSESSMENT:');
  console.log('=====================');
  
  const meetsMinimumRequirements = allCategories.length >= 35 && 
                                  allSubcategories.length >= 200 && 
                                  allProducts.length >= 900;
  
  const hasCompleteRelationships = categoriesWithSubcategories.size === allCategories.length &&
                                  categoriesWithProducts.size === allCategories.length &&
                                  subcategoriesWithProducts.size === allSubcategories.length;
  
  const hasNoOrphanedData = orphanedSubcategories.length === 0 && orphanedProducts.length === 0;
  
  console.log(`Categories requirement (35+): ${allCategories.length >= 35 ? '✅' : '❌'} ${allCategories.length}/35`);
  console.log(`Subcategories requirement (200+): ${allSubcategories.length >= 200 ? '✅' : '❌'} ${allSubcategories.length}/200`);
  console.log(`Products requirement (900+): ${allProducts.length >= 900 ? '✅' : '❌'} ${allProducts.length}/900`);
  console.log(`Complete relationships: ${hasCompleteRelationships ? '✅' : '❌'}`);
  console.log(`Data integrity: ${hasNoOrphanedData ? '✅' : '❌'}`);
  
  if (meetsMinimumRequirements && hasCompleteRelationships && hasNoOrphanedData) {
    console.log('\n🎉 SYSTEM STATUS: FULLY OPERATIONAL! ✨');
    console.log('=====================================');
    console.log('🛍️ Your e-commerce platform is ready for customers!');
    console.log('🚀 All 35 categories are populated');
    console.log('📦 All subcategories have products');
    console.log('💯 Complete product catalog available');
    console.log('✨ Featured products ready for homepage');
    console.log('🔗 All relationships properly linked');
  } else {
    console.log('\n⚠️  SYSTEM STATUS: PARTIALLY OPERATIONAL');
    console.log('========================================');
    if (!meetsMinimumRequirements) console.log('❌ Minimum data requirements not met');
    if (!hasCompleteRelationships) console.log('❌ Incomplete category-product relationships');
    if (!hasNoOrphanedData) console.log('❌ Data integrity issues detected');
    console.log('📝 Some improvements needed for full operation');
  }
});