const loki = require('lokijs');

// Initialize database
const db = new loki('users.db');

db.loadDatabase({}, () => {
  console.log('🔧 FILLING MISSING SUBCATEGORIES WITH PRODUCTS');
  console.log('=============================================\n');
  
  const categories = db.getCollection('categories');
  const subcategories = db.getCollection('subcategories');
  const products = db.getCollection('products');
  
  if (!categories || !subcategories || !products) {
    console.error('❌ Collections not found');
    return;
  }
  
  const allCategories = categories.find({});
  const allSubcategories = subcategories.find({});
  const allProducts = products.find({});
  
  // Find subcategories without products
  const subcategoriesWithProducts = new Set();
  allProducts.forEach(product => {
    if (product.subcategoryId) {
      subcategoriesWithProducts.add(product.subcategoryId);
    }
  });
  
  const subcategoriesWithoutProducts = allSubcategories.filter(sub => 
    !subcategoriesWithProducts.has(sub.id)
  );
  
  console.log(`📊 Found ${subcategoriesWithoutProducts.length} subcategories without products:`);
  subcategoriesWithoutProducts.forEach(sub => {
    const category = allCategories.find(cat => cat.id === sub.categoryId);
    console.log(`  - ${sub.name} (${category ? category.name : 'Unknown Category'})`);
  });
  
  if (subcategoriesWithoutProducts.length === 0) {
    console.log('✅ All subcategories already have products!');
    return;
  }
  
  console.log(`\n🎯 Adding products to ${subcategoriesWithoutProducts.length} subcategories...\n`);
  
  // Product templates
  const genericProducts = [
    'Premium Product', 'Professional Item', 'Quality Tool', 'Essential Equipment', 'Top-Rated Product',
    'Popular Choice', 'Best Seller', 'Customer Favorite', 'Recommended Item', 'Signature Product'
  ];
  
  const brands = ['ProMax', 'TechCore', 'UltraFit', 'SwiftPro', 'PrimeLine', 'EliteCore'];
  const adjectives = ['Premium', 'Professional', 'Advanced', 'Ultra', 'Super', 'Deluxe'];
  
  // Function to create slug from name
  function createSlug(name) {
    return name.toLowerCase()
      .replace(/[^a-z0-9 -]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim('-');
  }
  
  // Get next product ID
  let nextProductId = Math.max(...allProducts.map(p => p.id || 0)) + 1;
  
  let addedProducts = 0;
  
  // Add 2-3 products to each missing subcategory
  subcategoriesWithoutProducts.forEach(subcategory => {
    const category = allCategories.find(cat => cat.id === subcategory.categoryId);
    if (!category) return;
    
    const productsToAdd = Math.floor(Math.random() * 2) + 2; // 2-3 products
    
    for (let i = 0; i < productsToAdd; i++) {
      try {
        const template = genericProducts[Math.floor(Math.random() * genericProducts.length)];
        const brand = brands[Math.floor(Math.random() * brands.length)];
        const adjective = adjectives[Math.floor(Math.random() * adjectives.length)];
        
        const productName = `${brand} ${adjective} ${template}`;
        const basePrice = Math.round((Math.random() * 200 + 20) * 100) / 100;
        const comparePrice = Math.random() > 0.7 ? Math.round(basePrice * 1.3 * 100) / 100 : null;
        
        const product = {
          id: nextProductId++,
          name: productName,
          description: `High-quality ${productName.toLowerCase()} perfect for ${subcategory.name.toLowerCase()}. Designed with premium materials and excellent craftsmanship for ${category.name.toLowerCase()} enthusiasts.`,
          slug: createSlug(productName) + '-' + Math.random().toString(36).substr(2, 4),
          price: basePrice,
          comparePrice: comparePrice,
          categoryId: category.id,
          subcategoryId: subcategory.id,
          categoryName: category.name,
          subcategoryName: subcategory.name,
          sku: `${category.name.substr(0, 3).toUpperCase()}-${String(nextProductId).padStart(4, '0')}`,
          barcode: `${Math.floor(Math.random() * 9000000000000) + 1000000000000}`,
          stockQuantity: Math.floor(Math.random() * 50) + 10,
          lowStockAlert: 5,
          weight: Math.round((Math.random() * 3 + 0.1) * 100) / 100,
          dimensions: `${Math.floor(Math.random() * 25 + 5)} x ${Math.floor(Math.random() * 25 + 5)} x ${Math.floor(Math.random() * 15 + 2)} cm`,
          isActive: true,
          isFeatured: Math.random() > 0.85, // 15% chance of being featured
          tags: [
            subcategory.name.toLowerCase().replace(/\s+/g, '-'),
            category.name.toLowerCase().replace(/\s+/g, '-'),
            'premium', 'quality'
          ],
          imageUrl: `/images/products/${createSlug(subcategory.name)}.jpg`,
          metaTitle: `${productName} - ${subcategory.name}`,
          metaDescription: `Shop ${productName} in our ${subcategory.name} collection. Premium ${category.name.toLowerCase()} products with excellent quality.`,
          createdAt: new Date(Date.now() - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString(),
          updatedAt: new Date().toISOString()
        };
        
        products.insert(product);
        addedProducts++;
        
        console.log(`✅ Added "${productName}" to ${subcategory.name}`);
        
      } catch (error) {
        console.error(`❌ Error adding product to ${subcategory.name}:`, error);
      }
    }
  });
  
  // Save the database
  db.saveDatabase((err) => {
    if (err) {
      console.error('❌ Error saving database:', err);
    } else {
      console.log(`\n🎉 Successfully added ${addedProducts} products to missing subcategories!`);
      
      // Final verification
      const finalProducts = products.find({});
      const finalSubcategoriesWithProducts = new Set();
      finalProducts.forEach(product => {
        if (product.subcategoryId) {
          finalSubcategoriesWithProducts.add(product.subcategoryId);
        }
      });
      
      console.log('\n📊 FINAL VERIFICATION:');
      console.log('======================');
      console.log(`Total products: ${finalProducts.length}`);
      console.log(`Subcategories with products: ${finalSubcategoriesWithProducts.size}/${allSubcategories.length}`);
      console.log(`Featured products: ${finalProducts.filter(p => p.isFeatured).length}`);
      
      if (finalSubcategoriesWithProducts.size === allSubcategories.length) {
        console.log('\n🎉 SUCCESS: ALL subcategories now have products!');
      } else {
        const remaining = allSubcategories.length - finalSubcategoriesWithProducts.size;
        console.log(`\n⚠️  ${remaining} subcategories still need products`);
      }
      
      console.log('\n🛍️ PRODUCT COVERAGE COMPLETE! 🚀');
    }
  });
});