const fs = require('fs');
const path = require('path');

// Enhanced product generation with variable products
function createEnhancedProducts() {
  const dbPath = path.join(__dirname, 'users.db');
  
  if (!fs.existsSync(dbPath)) {
    console.error('Database file not found. Please run the basic setup first.');
    return;
  }

  const db = JSON.parse(fs.readFileSync(dbPath, 'utf8'));
  
  // Get existing categories and subcategories
  const categories = db.collections[1].data; // categories
  const subcategories = db.collections[2].data; // subcategories
  
  // Clear existing products
  db.collections[3].data = []; // products collection
  
  // Product attributes for variable products
  const productAttributes = {
    clothing: [
      { name: 'Size', values: ['XS', 'S', 'M', 'L', 'XL', 'XXL'] },
      { name: 'Color', values: ['Black', 'White', 'Red', 'Blue', 'Green', 'Gray'] }
    ],
    electronics: [
      { name: 'Storage', values: ['64GB', '128GB', '256GB', '512GB', '1TB'] },
      { name: 'Color', values: ['Black', 'White', 'Silver', 'Gold', 'Blue'] }
    ],
    footwear: [
      { name: 'Size', values: ['6', '7', '8', '9', '10', '11', '12'] },
      { name: 'Color', values: ['Black', 'White', 'Brown', 'Navy', 'Red'] }
    ],
    books: [],
    furniture: [
      { name: 'Material', values: ['Wood', 'Metal', 'Plastic', 'Glass'] },
      { name: 'Color', values: ['Brown', 'Black', 'White', 'Gray'] }
    ]
  };

  // Variable product categories (those that should have variants)
  const variableCategories = ['clothing', 'electronics', 'sports', 'home-garden'];
  
  const productNames = [
    'Premium', 'Elite', 'Professional', 'Advanced', 'Classic', 'Modern', 'Deluxe', 'Standard',
    'Ultimate', 'Essential', 'Superior', 'Quality', 'Signature', 'Exclusive', 'Perfect', 'Smart'
  ];

  const productTypes = [
    'Pro', 'Max', 'Plus', 'Air', 'Lite', 'Ultra', 'Core', 'Edge', 'Prime', 'Select'
  ];

  let productId = 1;

  // Create enhanced products for each subcategory
  subcategories.forEach(subcategory => {
    const category = categories.find(cat => cat.id === subcategory.categoryId);
    if (!category || !category.name) {
      console.log(`Skipping subcategory ${subcategory.name} - category not found or invalid`);
      return;
    }

    const isVariableCategory = variableCategories.includes(category.slug);
    const productsPerSubcategory = Math.floor(Math.random() * 8) + 5; // 5-12 products per subcategory

    for (let i = 0; i < productsPerSubcategory; i++) {
      const baseName = productNames[Math.floor(Math.random() * productNames.length)];
      const productType = productTypes[Math.floor(Math.random() * productTypes.length)];
      const productName = `${baseName} ${productType} ${subcategory.name.split(' ')[0]}`;
      
      const slug = productName.toLowerCase()
        .replace(/[^a-z0-9\s-]/g, '')
        .replace(/\s+/g, '-')
        .replace(/-+/g, '-') + `-${productId.toString().padStart(3, '0')}`;

      const basePrice = Math.floor(Math.random() * 800) + 20; // $20-$820
      const hasComparePrice = Math.random() > 0.7;
      const comparePrice = hasComparePrice ? basePrice + Math.floor(Math.random() * 200) + 50 : undefined;

      // Decide if this should be a variable product (30% chance for variable categories)
      const shouldBeVariable = isVariableCategory && Math.random() > 0.7;

      if (shouldBeVariable) {
        // Create variable product
        const categoryAttrs = productAttributes[category.slug] || productAttributes.clothing;
        const selectedAttrs = categoryAttrs.slice(0, Math.random() > 0.5 ? 2 : 1);

        // Generate variants
        const variants = [];
        const generateVariants = (attrs, index = 0, currentCombination = {}) => {
          if (index === attrs.length) {
            const variantPrice = basePrice + Math.floor(Math.random() * 100) - 50; // ±$50 variation
            const variantId = `var-${productId}-${variants.length + 1}`;
            
            variants.push({
              id: variantId,
              sku: `SKU-${variantId.toUpperCase()}`,
              price: Math.max(variantPrice, 10),
              comparePrice: hasComparePrice ? (variantPrice + Math.floor(Math.random() * 100) + 30) : undefined,
              stockQuantity: Math.floor(Math.random() * 50) + 5,
              attributes: { ...currentCombination },
              imageUrl: `/images/products/${slug}-${variants.length + 1}.jpg`,
              isActive: true
            });
            return;
          }

          const attr = attrs[index];
          attr.values.forEach(value => {
            generateVariants(attrs, index + 1, { ...currentCombination, [attr.name]: value });
          });
        };

        generateVariants(selectedAttrs);

        // Calculate price range
        const prices = variants.map(v => v.price);
        const priceRange = {
          min: Math.min(...prices),
          max: Math.max(...prices)
        };

        const variableProduct = {
          $loki: productId,
          id: productId.toString(),
          name: productName,
          description: `High-quality ${productName.toLowerCase()} with multiple options. Perfect for ${category.name.toLowerCase()} enthusiasts. Choose from various ${selectedAttrs.map(a => a.name.toLowerCase()).join(' and ')} combinations.`,
          categoryId: category.id,
          subcategoryId: subcategory.id,
          slug: slug,
          isActive: true,
          isFeatured: Math.random() > 0.8,
          images: [
            `/images/products/${slug}-main.jpg`,
            `/images/products/${slug}-alt1.jpg`,
            `/images/products/${slug}-alt2.jpg`
          ],
          createdAt: new Date().toISOString(),
          type: 'variable',
          attributes: selectedAttrs,
          variants: variants,
          priceRange: priceRange
        };

        db.collections[3].data.push(variableProduct);
      } else {
        // Create simple product
        const simpleProduct = {
          $loki: productId,
          id: productId.toString(),
          name: productName,
          description: `Premium ${productName.toLowerCase()} designed for ${category.name.toLowerCase()}. High-quality materials and excellent craftsmanship make this an essential item.`,
          price: basePrice,
          comparePrice: comparePrice,
          categoryId: category.id,
          subcategoryId: subcategory.id,
          imageUrl: `/images/products/${slug}.jpg`,
          slug: slug,
          isActive: true,
          isFeatured: Math.random() > 0.8,
          stockQuantity: Math.floor(Math.random() * 100) + 10,
          sku: `SKU-${productId.toString().padStart(6, '0')}`,
          images: [`/images/products/${slug}.jpg`],
          createdAt: new Date().toISOString(),
          type: 'simple'
        };

        db.collections[3].data.push(simpleProduct);
      }

      productId++;
    }
  });

  // Update collection metadata
  db.collections[3].maxId = productId - 1;

  // Write updated database
  fs.writeFileSync(dbPath, JSON.stringify(db, null, 2));
  
  const totalProducts = db.collections[3].data.length;
  const simpleProducts = db.collections[3].data.filter(p => p.type === 'simple').length;
  const variableProducts = db.collections[3].data.filter(p => p.type === 'variable').length;
  const totalVariants = db.collections[3].data
    .filter(p => p.type === 'variable')
    .reduce((sum, p) => sum + p.variants.length, 0);

  console.log('✅ Enhanced product system created successfully!');
  console.log('📊 Product Statistics:');
  console.log(`   📦 Total Products: ${totalProducts}`);
  console.log(`   🔧 Simple Products: ${simpleProducts}`);
  console.log(`   🎛️  Variable Products: ${variableProducts}`);
  console.log(`   📋 Total Variants: ${totalVariants}`);
  console.log('🎯 Product types distribution:');
  console.log(`   Simple: ${Math.round((simpleProducts / totalProducts) * 100)}%`);
  console.log(`   Variable: ${Math.round((variableProducts / totalProducts) * 100)}%`);
}

createEnhancedProducts();