const fs = require('fs');
const path = require('path');

// Read the existing database file
const dbPath = path.join(__dirname, 'users.db');
let dbData;

try {
  const rawData = fs.readFileSync(dbPath, 'utf8');
  dbData = JSON.parse(rawData);
} catch (error) {
  console.error('Error reading database:', error);
  process.exit(1);
}

// Find or create subcategories collection
let subcategoriesCollection = dbData.collections.find(c => c.name === 'subcategories');
if (!subcategoriesCollection) {
  subcategoriesCollection = {
    name: 'subcategories',
    data: [],
    idIndex: 1,
    binaryIndices: {},
    constraints: {},
    uniqueNames: [],
    transforms: {},
    objType: 'subcategories',
    dirty: true,
    cachedIndex: null,
    cachedBinaryIndex: null,
    cachedData: null,
    adaptiveBinaryIndices: true,
    transactional: false,
    cloneObjects: false,
    cloneMethod: 'parse-stringify',
    asyncListeners: false,
    disableMeta: false,
    disableChangesApi: true,
    disableDeltaChangesApi: true,
    autoupdate: false,
    serializableIndices: true,
    disableFreeze: true,
    ttl: null,
    maxId: 0,
    DynamicViews: [],
    events: {},
    changes: [],
    dirtyIds: []
  };
  dbData.collections.push(subcategoriesCollection);
}

// Get categories collection to create realistic subcategories
const categoriesCollection = dbData.collections.find(c => c.name === 'categories');
const categories = categoriesCollection ? categoriesCollection.data : [];

console.log('Found categories:', categories.map(c => c.name));

// Define subcategories for different categories
const subcategoriesData = [
  // Electronics subcategories
  { categoryName: 'Electronics', name: 'Smartphones', description: 'Mobile phones and accessories', icon: '📱' },
  { categoryName: 'Electronics', name: 'Laptops', description: 'Portable computers and laptops', icon: '💻' },
  { categoryName: 'Electronics', name: 'Tablets', description: 'Tablet computers and e-readers', icon: '📱' },
  { categoryName: 'Electronics', name: 'Audio', description: 'Headphones, speakers, and audio equipment', icon: '🎧' },
  { categoryName: 'Electronics', name: 'Gaming', description: 'Gaming consoles and accessories', icon: '🎮' },
  
  // Clothing subcategories
  { categoryName: 'Clothing', name: 'Men\'s Clothing', description: 'Clothing for men', icon: '👔' },
  { categoryName: 'Clothing', name: 'Women\'s Clothing', description: 'Clothing for women', icon: '👗' },
  { categoryName: 'Clothing', name: 'Kids Clothing', description: 'Clothing for children', icon: '👶' },
  { categoryName: 'Clothing', name: 'Shoes', description: 'Footwear for all ages', icon: '👟' },
  { categoryName: 'Clothing', name: 'Accessories', description: 'Fashion accessories', icon: '👜' },
  
  // Home & Garden subcategories
  { categoryName: 'Home & Garden', name: 'Furniture', description: 'Home and office furniture', icon: '🪑' },
  { categoryName: 'Home & Garden', name: 'Kitchen', description: 'Kitchen appliances and utensils', icon: '🍳' },
  { categoryName: 'Home & Garden', name: 'Bedroom', description: 'Bedroom furniture and decor', icon: '🛏️' },
  { categoryName: 'Home & Garden', name: 'Garden Tools', description: 'Gardening equipment and tools', icon: '🌱' },
  { categoryName: 'Home & Garden', name: 'Home Decor', description: 'Decorative items for home', icon: '🖼️' },
  
  // Sports subcategories
  { categoryName: 'Sports', name: 'Fitness Equipment', description: 'Exercise and fitness gear', icon: '🏋️' },
  { categoryName: 'Sports', name: 'Outdoor Sports', description: 'Equipment for outdoor activities', icon: '🏕️' },
  { categoryName: 'Sports', name: 'Team Sports', description: 'Equipment for team sports', icon: '⚽' },
  { categoryName: 'Sports', name: 'Water Sports', description: 'Equipment for water activities', icon: '🏄' },
  { categoryName: 'Sports', name: 'Winter Sports', description: 'Equipment for winter sports', icon: '⛷️' },
  
  // Books subcategories
  { categoryName: 'Books', name: 'Fiction', description: 'Novels and fictional literature', icon: '📚' },
  { categoryName: 'Books', name: 'Non-Fiction', description: 'Educational and informational books', icon: '📖' },
  { categoryName: 'Books', name: 'Children\'s Books', description: 'Books for children and young adults', icon: '📕' },
  { categoryName: 'Books', name: 'Textbooks', description: 'Academic and educational textbooks', icon: '📘' },
  { categoryName: 'Books', name: 'Comics & Manga', description: 'Comic books and manga', icon: '📗' }
];

console.log('Creating subcategories...');

let subcategoryCount = 0;

subcategoriesData.forEach((subcat, index) => {
  // Find the parent category
  const parentCategory = categories.find(c => c.name === subcat.categoryName);
  
  if (parentCategory) {
    const subcategory = {
      id: `SUBCAT-${Date.now()}-${(subcategoryCount + 1).toString().padStart(3, '0')}`,
      name: subcat.name,
      description: subcat.description,
      slug: subcat.name.toLowerCase()
        .replace(/[^a-z0-9 -]/g, '')
        .replace(/\s+/g, '-')
        .replace(/-+/g, '-')
        .trim(),
      categoryId: parentCategory.id,
      icon: subcat.icon,
      isActive: Math.random() > 0.1, // 90% active
      sortOrder: index + 1,
      createdAt: new Date(Date.now() - Math.random() * 365 * 24 * 60 * 60 * 1000).toISOString(), // Random date within last year
      updatedAt: new Date().toISOString(),
      meta: {
        revision: 0,
        created: Date.now(),
        version: 0
      },
      $loki: subcategoryCount + 1
    };
    
    subcategoriesCollection.data.push(subcategory);
    subcategoryCount++;
    
    console.log(`✅ Added: ${subcat.name} → ${subcat.categoryName}`);
  } else {
    console.log(`❌ Category "${subcat.categoryName}" not found for subcategory "${subcat.name}"`);
  }
});

// Update collection metadata
subcategoriesCollection.maxId = subcategoryCount;
subcategoriesCollection.dirty = true;

// Write the updated database back to file
try {
  fs.writeFileSync(dbPath, JSON.stringify(dbData, null, 2));
  console.log(`\n🎉 Successfully added ${subcategoryCount} subcategories to the database!`);
  
  // Show some stats
  const activeSubcategories = subcategoriesCollection.data.filter(sc => sc.isActive).length;
  const categoriesWithSubcategories = new Set(subcategoriesCollection.data.map(sc => sc.categoryId)).size;
  
  console.log(`📊 Stats:`);
  console.log(`   Total subcategories: ${subcategoryCount}`);
  console.log(`   Active subcategories: ${activeSubcategories}`);
  console.log(`   Categories with subcategories: ${categoriesWithSubcategories}`);
  
} catch (error) {
  console.error('❌ Error writing to database:', error);
  process.exit(1);
}