const fs = require('fs');

// Product categories and subcategories for variety
const categories = [
  { name: 'Electronics', subcategories: ['Smartphones', 'Laptops', 'Tablets', 'Accessories', 'Gaming'] },
  { name: 'Clothing', subcategories: ['Men\'s Wear', 'Women\'s Wear', 'Kids Wear', 'Shoes', 'Accessories'] },
  { name: 'Home & Garden', subcategories: ['Furniture', 'Kitchen', 'Bathroom', 'Garden', 'Decor'] },
  { name: 'Sports', subcategories: ['Fitness', 'Outdoor', 'Team Sports', 'Water Sports', 'Winter Sports'] },
  { name: 'Books', subcategories: ['Fiction', 'Non-Fiction', 'Educational', 'Children\'s', 'Comics'] },
  { name: 'Beauty', subcategories: ['Skincare', 'Makeup', 'Hair Care', 'Fragrances', 'Tools'] },
  { name: 'Food', subcategories: ['Snacks', 'Beverages', 'Organic', 'International', 'Gourmet'] },
  { name: 'Toys', subcategories: ['Educational', 'Action Figures', 'Board Games', 'Outdoor', 'Electronic'] }
];

// Product name prefixes and suffixes
const prefixes = ['Premium', 'Deluxe', 'Pro', 'Ultra', 'Smart', 'Advanced', 'Classic', 'Modern', 'Eco', 'Luxury'];
const suffixes = ['Plus', 'Max', 'Elite', 'Supreme', 'Special', 'Limited', 'Enhanced', 'Professional', 'Standard', 'Essential'];

// Generate random product name
function generateProductName(category, subcategory) {
  const prefix = prefixes[Math.floor(Math.random() * prefixes.length)];
  const suffix = suffixes[Math.floor(Math.random() * suffixes.length)];
  const number = Math.floor(Math.random() * 9999) + 1;
  
  return `${prefix} ${subcategory} ${suffix} ${number}`;
}

// Generate random description
function generateDescription(productName) {
  const descriptions = [
    `High-quality ${productName.toLowerCase()} designed for modern lifestyle. Perfect for daily use.`,
    `Experience the best with our ${productName.toLowerCase()}. Made with premium materials.`,
    `Top-rated ${productName.toLowerCase()} featuring innovative design and superior performance.`,
    `Professional-grade ${productName.toLowerCase()} built to last. Excellent value for money.`,
    `Revolutionary ${productName.toLowerCase()} that combines style and functionality seamlessly.`
  ];
  
  return descriptions[Math.floor(Math.random() * descriptions.length)];
}

// Generate random price
function generatePrice() {
  const basePrice = Math.random() * 2000 + 10; // $10 to $2010
  return Math.round(basePrice * 100) / 100; // Round to 2 decimal places
}

// Generate random SKU
function generateSKU() {
  const letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
  const numbers = '0123456789';
  let sku = '';
  
  // 3 letters
  for (let i = 0; i < 3; i++) {
    sku += letters.charAt(Math.floor(Math.random() * letters.length));
  }
  
  // 4 numbers
  for (let i = 0; i < 4; i++) {
    sku += numbers.charAt(Math.floor(Math.random() * numbers.length));
  }
  
  return sku;
}

// Generate slug from name
function generateSlug(name) {
  return name.toLowerCase()
    .replace(/[^a-z0-9 -]/g, '')
    .replace(/\s+/g, '-')
    .replace(/-+/g, '-')
    .trim();
}

// Main function to create dummy products
async function createDummyProducts() {
  console.log('🚀 Starting to create 5000 dummy products...');
  
  try {
    // Read the current database
    const dbPath = 'users.db';
    let data;
    
    if (fs.existsSync(dbPath)) {
      const dbContent = fs.readFileSync(dbPath, 'utf8');
      data = JSON.parse(dbContent);
    } else {
      console.error('❌ Database file not found!');
      return;
    }
    
    // Find or create products collection
    let productsCollection = data.collections.find(c => c.name === 'products');
    if (!productsCollection) {
      productsCollection = {
        name: 'products',
        data: [],
        idIndex: [],
        binaryIndices: {},
        constraints: { unique: {}, exact: {} },
        uniqueNames: [],
        transforms: {},
        objType: 'products',
        dirty: true,
        cachedIndex: null,
        cachedBinaryIndex: null,
        cachedData: null,
        adaptiveBinaryIndices: true,
        transactional: false,
        cloneObjects: true,
        cloneMethod: 'parse-stringify',
        asyncListeners: false,
        disableMeta: false,
        disableChangesApi: true,
        disableDeltaChangesApi: true,
        autoupdate: false,
        serializableIndices: true,
        disableFreeze: true,
        ttl: null,
        maxId: 0,
        DynamicViews: [],
        events: {
          'insert': [],
          'update': [],
          'pre-insert': [],
          'pre-update': [],
          'close': [],
          'flushbuffer': [],
          'error': [],
          'delete': [],
          'warning': []
        },
        changes: [],
        dirtyIds: []
      };
      data.collections.push(productsCollection);
    }
    
    // Get current max ID
    let maxId = 0;
    if (productsCollection.data.length > 0) {
      maxId = Math.max(...productsCollection.data.map(p => p.id || 0));
    }
    
    const currentDate = new Date().toISOString();
    const newProducts = [];
    
    console.log('📦 Generating products...');
    
    for (let i = 0; i < 5000; i++) {
      // Select random category and subcategory
      const category = categories[Math.floor(Math.random() * categories.length)];
      const subcategory = category.subcategories[Math.floor(Math.random() * category.subcategories.length)];
      
      const productName = generateProductName(category.name, subcategory);
      const price = generatePrice();
      const stock = Math.floor(Math.random() * 500) + 10; // 10-509 stock
      const sku = generateSKU();
      
      const product = {
        id: maxId + i + 1,
        $loki: maxId + i + 1,
        name: productName,
        description: generateDescription(productName),
        price: price,
        originalPrice: Math.round((price * (1 + Math.random() * 0.5)) * 100) / 100, // 0-50% markup
        sku: sku,
        slug: generateSlug(productName),
        category: category.name,
        subcategory: subcategory,
        stock: stock,
        images: [
          `https://picsum.photos/400/400?random=${i + 1}`,
          `https://picsum.photos/400/400?random=${i + 5001}`,
          `https://picsum.photos/400/400?random=${i + 10001}`
        ],
        status: Math.random() > 0.1 ? 'active' : 'inactive', // 90% active
        featured: Math.random() > 0.8, // 20% featured
        tags: [category.name.toLowerCase(), subcategory.toLowerCase(), 'new'],
        weight: Math.round((Math.random() * 10 + 0.1) * 100) / 100, // 0.1-10.1 kg
        dimensions: {
          length: Math.round((Math.random() * 50 + 5) * 10) / 10, // 5-55 cm
          width: Math.round((Math.random() * 50 + 5) * 10) / 10,
          height: Math.round((Math.random() * 50 + 5) * 10) / 10
        },
        rating: Math.round((Math.random() * 2 + 3) * 10) / 10, // 3.0-5.0 rating
        reviewCount: Math.floor(Math.random() * 1000),
        meta: {
          created: currentDate,
          revision: 0
        },
        createdAt: currentDate,
        updatedAt: currentDate
      };
      
      newProducts.push(product);
      
      // Show progress
      if ((i + 1) % 500 === 0) {
        console.log(`✅ Generated ${i + 1}/5000 products...`);
      }
    }
    
    // Add products to collection
    productsCollection.data.push(...newProducts);
    productsCollection.maxId = maxId + 5000;
    
    // Update idIndex
    productsCollection.idIndex = productsCollection.data.map((_, index) => index);
    
    // Write back to database
    fs.writeFileSync(dbPath, JSON.stringify(data, null, 2));
    
    console.log('🎉 Successfully created 5000 dummy products!');
    console.log(`📊 Total products in database: ${productsCollection.data.length}`);
    console.log('');
    console.log('Product categories distribution:');
    
    // Show distribution
    const distribution = {};
    newProducts.forEach(product => {
      distribution[product.category] = (distribution[product.category] || 0) + 1;
    });
    
    Object.entries(distribution).forEach(([category, count]) => {
      console.log(`  ${category}: ${count} products`);
    });
    
  } catch (error) {
    console.error('❌ Error creating dummy products:', error);
  }
}

// Run the script
createDummyProducts();