const fs = require('fs');
const path = require('path');

// Read the existing database file
const dbPath = path.join(__dirname, 'users.db');
let dbData;

try {
  const rawData = fs.readFileSync(dbPath, 'utf8');
  dbData = JSON.parse(rawData);
} catch (error) {
  console.error('Error reading database:', error);
  process.exit(1);
}

// Find or create customers collection
let customersCollection = dbData.collections.find(c => c.name === 'customers');
if (!customersCollection) {
  customersCollection = {
    name: 'customers',
    data: [],
    idIndex: 1,
    binaryIndices: {},
    constraints: {},
    uniqueNames: [],
    transforms: {},
    objType: 'customers',
    dirty: true,
    cachedIndex: null,
    cachedBinaryIndex: null,
    cachedData: null,
    adaptiveBinaryIndices: true,
    transactional: false,
    cloneObjects: false,
    cloneMethod: 'parse-stringify',
    asyncListeners: false,
    disableMeta: false,
    disableChangesApi: true,
    disableDeltaChangesApi: true,
    autoupdate: false,
    serializableIndices: true,
    disableFreeze: true,
    ttl: null,
    maxId: 500,
    DynamicViews: [],
    events: {},
    changes: [],
    dirtyIds: []
  };
  dbData.collections.push(customersCollection);
}

// Generate 500 dummy customers
const firstNames = ['John', 'Jane', 'Michael', 'Sarah', 'David', 'Emily', 'Robert', 'Jessica', 'William', 'Ashley', 'James', 'Amanda', 'Christopher', 'Stephanie', 'Daniel', 'Jennifer', 'Matthew', 'Elizabeth', 'Anthony', 'Megan', 'Mark', 'Lauren', 'Donald', 'Nicole', 'Steven', 'Samantha', 'Andrew', 'Rachel', 'Kenneth', 'Amy', 'Paul', 'Heather', 'Joshua', 'Michelle', 'Kevin', 'Kimberly', 'Brian', 'Lisa', 'George', 'Angela'];
const lastNames = ['Smith', 'Johnson', 'Williams', 'Brown', 'Jones', 'Garcia', 'Miller', 'Davis', 'Rodriguez', 'Martinez', 'Hernandez', 'Lopez', 'Gonzalez', 'Wilson', 'Anderson', 'Thomas', 'Taylor', 'Moore', 'Jackson', 'Martin', 'Lee', 'Perez', 'Thompson', 'White', 'Harris', 'Sanchez', 'Clark', 'Ramirez', 'Lewis', 'Robinson', 'Walker', 'Young', 'Allen', 'King', 'Wright', 'Scott', 'Torres', 'Nguyen', 'Hill', 'Flores'];
const cities = ['New York', 'Los Angeles', 'Chicago', 'Houston', 'Phoenix', 'Philadelphia', 'San Antonio', 'San Diego', 'Dallas', 'San Jose', 'Austin', 'Jacksonville', 'San Francisco', 'Columbus', 'Charlotte', 'Fort Worth', 'Indianapolis', 'Seattle', 'Denver', 'Boston'];
const states = ['NY', 'CA', 'IL', 'TX', 'AZ', 'PA', 'FL', 'WA', 'OH', 'NC', 'IN', 'CO', 'MA', 'GA', 'VA', 'MI', 'NV', 'OR', 'MD', 'TN'];

// Helper function to generate random date within last 2 years
const getRandomDate = (start, end) => {
  return new Date(start.getTime() + Math.random() * (end.getTime() - start.getTime()));
};

// Helper function to generate random phone number
const getRandomPhone = () => {
  const area = Math.floor(Math.random() * 800) + 200;
  const exchange = Math.floor(Math.random() * 800) + 200;
  const number = Math.floor(Math.random() * 10000);
  return `(${area}) ${exchange}-${number.toString().padStart(4, '0')}`;
};

// Helper function to generate random zip code
const getRandomZip = () => {
  return Math.floor(Math.random() * 90000) + 10000;
};

console.log('Generating 500 dummy customers...');

const startDate = new Date(2022, 0, 1);
const endDate = new Date();

for (let i = 1; i <= 500; i++) {
  const firstName = firstNames[Math.floor(Math.random() * firstNames.length)];
  const lastName = lastNames[Math.floor(Math.random() * lastNames.length)];
  const email = `${firstName.toLowerCase()}.${lastName.toLowerCase()}${i}@example.com`;
  const createdAt = getRandomDate(startDate, endDate);
  const lastOrderDate = Math.random() > 0.3 ? getRandomDate(createdAt, endDate) : null;
  const totalOrders = Math.floor(Math.random() * 25);
  const totalSpent = totalOrders > 0 ? Math.floor(Math.random() * 5000) + 50 : 0;
  const status = Math.random() > 0.1 ? 'active' : 'inactive';
  
  const city = cities[Math.floor(Math.random() * cities.length)];
  const state = states[Math.floor(Math.random() * states.length)];
  
  const customer = {
    id: `CUST-${i.toString().padStart(4, '0')}`,
    firstName,
    lastName,
    email,
    phone: Math.random() > 0.2 ? getRandomPhone() : null,
    address: Math.random() > 0.1 ? {
      street: `${Math.floor(Math.random() * 9999) + 1} ${['Main', 'Oak', 'Pine', 'Maple', 'Cedar', 'Elm', 'Park', 'First', 'Second', 'Third'][Math.floor(Math.random() * 10)]} St`,
      city,
      state,
      zipCode: getRandomZip().toString(),
      country: 'USA'
    } : null,
    totalOrders,
    totalSpent,
    lastOrderDate: lastOrderDate ? lastOrderDate.toISOString() : null,
    status,
    createdAt: createdAt.toISOString(),
    updatedAt: createdAt.toISOString(),
    meta: {
      revision: 0,
      created: Date.now(),
      version: 0
    },
    $loki: i
  };
  
  customersCollection.data.push(customer);
}

// Update collection metadata
customersCollection.maxId = 500;
customersCollection.dirty = true;

// Update database metadata
dbData.databaseVersion = 1.5;
dbData.engineVersion = 1.5;
dbData.autosave = true;
dbData.autosaveInterval = 5000;
dbData.options = {};
dbData.persistenceMethod = 'fs';
dbData.persistenceAdapter = null;
dbData.verbose = false;
dbData.events = {};
dbData.changes = [];

// Write the updated database back to file
try {
  fs.writeFileSync(dbPath, JSON.stringify(dbData, null, 2));
  console.log('✅ Successfully added 500 dummy customers to the database!');
  console.log(`📊 Total customers in database: ${customersCollection.data.length}`);
  
  // Show some stats
  const activeCustomers = customersCollection.data.filter(c => c.status === 'active').length;
  const totalRevenue = customersCollection.data.reduce((sum, c) => sum + c.totalSpent, 0);
  const totalOrders = customersCollection.data.reduce((sum, c) => sum + c.totalOrders, 0);
  
  console.log(`✅ Active customers: ${activeCustomers}`);
  console.log(`💰 Total revenue: $${totalRevenue.toLocaleString()}`);
  console.log(`📦 Total orders: ${totalOrders.toLocaleString()}`);
  
} catch (error) {
  console.error('❌ Error writing to database:', error);
  process.exit(1);
}