const fs = require('fs');
const path = require('path');

// Read the existing database file
const dbPath = path.join(__dirname, 'users.db');
let dbData;

try {
  const rawData = fs.readFileSync(dbPath, 'utf8');
  dbData = JSON.parse(rawData);
} catch (error) {
  console.error('Error reading database:', error);
  process.exit(1);
}

// Find or create collections
let categoriesCollection = dbData.collections.find(c => c.name === 'categories');
let subcategoriesCollection = dbData.collections.find(c => c.name === 'subcategories');

if (!categoriesCollection) {
  categoriesCollection = {
    name: 'categories',
    data: [],
    idIndex: 1,
    binaryIndices: {},
    constraints: {},
    uniqueNames: [],
    transforms: {},
    objType: 'categories',
    dirty: true,
    cachedIndex: null,
    cachedBinaryIndex: null,
    cachedData: null,
    adaptiveBinaryIndices: true,
    transactional: false,
    cloneObjects: false,
    cloneMethod: 'parse-stringify',
    asyncListeners: false,
    disableMeta: false,
    disableChangesApi: true,
    disableDeltaChangesApi: true,
    autoupdate: false,
    serializableIndices: true,
    disableFreeze: true,
    ttl: null,
    maxId: 0,
    DynamicViews: [],
    events: {},
    changes: [],
    dirtyIds: []
  };
  dbData.collections.push(categoriesCollection);
}

if (!subcategoriesCollection) {
  subcategoriesCollection = {
    name: 'subcategories',
    data: [],
    idIndex: 1,
    binaryIndices: {},
    constraints: {},
    uniqueNames: [],
    transforms: {},
    objType: 'subcategories',
    dirty: true,
    cachedIndex: null,
    cachedBinaryIndex: null,
    cachedData: null,
    adaptiveBinaryIndices: true,
    transactional: false,
    cloneObjects: false,
    cloneMethod: 'parse-stringify',
    asyncListeners: false,
    disableMeta: false,
    disableChangesApi: true,
    disableDeltaChangesApi: true,
    autoupdate: false,
    serializableIndices: true,
    disableFreeze: true,
    ttl: null,
    maxId: 0,
    DynamicViews: [],
    events: {},
    changes: [],
    dirtyIds: []
  };
  dbData.collections.push(subcategoriesCollection);
}

// Clear existing data to start fresh
categoriesCollection.data = [];
subcategoriesCollection.data = [];

console.log('🏗️  Creating comprehensive categories and subcategories...');

// Define comprehensive categories and their subcategories
const categoriesWithSubcategories = [
  {
    name: 'Electronics',
    description: 'Electronic devices, gadgets, and accessories',
    icon: '📱',
    subcategories: [
      { name: 'Smartphones', description: 'Mobile phones and accessories', icon: '📱' },
      { name: 'Laptops & Computers', description: 'Portable and desktop computers', icon: '💻' },
      { name: 'Tablets', description: 'Tablet computers and e-readers', icon: '📱' },
      { name: 'Audio & Headphones', description: 'Headphones, earbuds, speakers', icon: '🎧' },
      { name: 'Cameras', description: 'Digital cameras and photography equipment', icon: '📷' },
      { name: 'Gaming', description: 'Gaming consoles, accessories, and games', icon: '🎮' },
      { name: 'Smart Home', description: 'Smart home devices and automation', icon: '🏠' },
      { name: 'Wearables', description: 'Smartwatches and fitness trackers', icon: '⌚' },
      { name: 'TV & Entertainment', description: 'Televisions and streaming devices', icon: '📺' },
      { name: 'Accessories', description: 'Cables, chargers, and other accessories', icon: '🔌' }
    ]
  },
  {
    name: 'Fashion & Clothing',
    description: 'Clothing, footwear, and fashion accessories',
    icon: '👗',
    subcategories: [
      { name: 'Men\'s Clothing', description: 'Clothing and apparel for men', icon: '👔' },
      { name: 'Women\'s Clothing', description: 'Clothing and apparel for women', icon: '👗' },
      { name: 'Kids & Baby', description: 'Clothing for children and babies', icon: '👶' },
      { name: 'Shoes & Footwear', description: 'Shoes, boots, sneakers, and sandals', icon: '👟' },
      { name: 'Bags & Luggage', description: 'Handbags, backpacks, and travel luggage', icon: '👜' },
      { name: 'Jewelry & Watches', description: 'Jewelry, watches, and accessories', icon: '💍' },
      { name: 'Sunglasses', description: 'Sunglasses and eyewear', icon: '🕶️' },
      { name: 'Hats & Caps', description: 'Hats, caps, and head accessories', icon: '🧢' },
      { name: 'Belts & Wallets', description: 'Belts, wallets, and leather goods', icon: '👛' },
      { name: 'Underwear & Socks', description: 'Undergarments and hosiery', icon: '🧦' }
    ]
  },
  {
    name: 'Home & Garden',
    description: 'Home improvement, furniture, and garden supplies',
    icon: '🏡',
    subcategories: [
      { name: 'Furniture', description: 'Home and office furniture', icon: '🪑' },
      { name: 'Kitchen & Dining', description: 'Kitchen appliances and dining essentials', icon: '🍽️' },
      { name: 'Bedroom', description: 'Bedroom furniture and bedding', icon: '🛏️' },
      { name: 'Bathroom', description: 'Bathroom fixtures and accessories', icon: '🚿' },
      { name: 'Living Room', description: 'Living room furniture and decor', icon: '🛋️' },
      { name: 'Home Decor', description: 'Decorative items and artwork', icon: '🖼️' },
      { name: 'Lighting', description: 'Lamps, fixtures, and lighting solutions', icon: '💡' },
      { name: 'Garden & Outdoor', description: 'Gardening tools and outdoor furniture', icon: '🌱' },
      { name: 'Tools & Hardware', description: 'Hand tools and hardware supplies', icon: '🔨' },
      { name: 'Storage & Organization', description: 'Storage solutions and organizers', icon: '📦' }
    ]
  },
  {
    name: 'Sports & Outdoors',
    description: 'Sports equipment, outdoor gear, and fitness products',
    icon: '⚽',
    subcategories: [
      { name: 'Fitness Equipment', description: 'Exercise machines and workout gear', icon: '🏋️' },
      { name: 'Team Sports', description: 'Equipment for football, basketball, soccer', icon: '🏀' },
      { name: 'Outdoor Recreation', description: 'Camping, hiking, and outdoor activities', icon: '🏕️' },
      { name: 'Water Sports', description: 'Swimming, surfing, and water activities', icon: '🏄' },
      { name: 'Winter Sports', description: 'Skiing, snowboarding, and winter gear', icon: '⛷️' },
      { name: 'Cycling', description: 'Bicycles, cycling gear, and accessories', icon: '🚴' },
      { name: 'Running', description: 'Running shoes, apparel, and accessories', icon: '🏃' },
      { name: 'Golf', description: 'Golf clubs, balls, and accessories', icon: '⛳' },
      { name: 'Tennis', description: 'Tennis rackets, balls, and court equipment', icon: '🎾' },
      { name: 'Yoga & Pilates', description: 'Yoga mats, blocks, and wellness products', icon: '🧘' }
    ]
  },
  {
    name: 'Books & Media',
    description: 'Books, magazines, movies, and digital media',
    icon: '📚',
    subcategories: [
      { name: 'Fiction Books', description: 'Novels, short stories, and fiction literature', icon: '📖' },
      { name: 'Non-Fiction', description: 'Biographies, self-help, and educational books', icon: '📘' },
      { name: 'Children\'s Books', description: 'Books for kids and young adults', icon: '📕' },
      { name: 'Textbooks', description: 'Academic and educational textbooks', icon: '📓' },
      { name: 'Comics & Graphic Novels', description: 'Comic books, manga, and graphic novels', icon: '📗' },
      { name: 'Magazines', description: 'Periodicals and magazine subscriptions', icon: '📰' },
      { name: 'Movies & TV', description: 'DVDs, Blu-rays, and digital movies', icon: '🎬' },
      { name: 'Music', description: 'CDs, vinyl records, and digital music', icon: '🎵' },
      { name: 'Video Games', description: 'Console and PC video games', icon: '🕹️' },
      { name: 'E-books & Audiobooks', description: 'Digital books and audio content', icon: '🎧' }
    ]
  },
  {
    name: 'Health & Beauty',
    description: 'Health products, cosmetics, and personal care items',
    icon: '💄',
    subcategories: [
      { name: 'Skincare', description: 'Face care, moisturizers, and treatments', icon: '🧴' },
      { name: 'Makeup & Cosmetics', description: 'Foundation, lipstick, and beauty products', icon: '💄' },
      { name: 'Hair Care', description: 'Shampoo, conditioner, and styling products', icon: '💇' },
      { name: 'Fragrances', description: 'Perfumes and cologne', icon: '🌸' },
      { name: 'Personal Care', description: 'Toothcare, deodorants, and hygiene products', icon: '🪥' },
      { name: 'Health Supplements', description: 'Vitamins, minerals, and supplements', icon: '💊' },
      { name: 'Medical Supplies', description: 'First aid and medical equipment', icon: '🩹' },
      { name: 'Wellness', description: 'Aromatherapy, meditation, and wellness products', icon: '🕯️' },
      { name: 'Baby Care', description: 'Baby skincare and hygiene products', icon: '🍼' },
      { name: 'Men\'s Grooming', description: 'Shaving, beard care, and men\'s products', icon: '🪒' }
    ]
  },
  {
    name: 'Automotive',
    description: 'Car parts, accessories, and automotive supplies',
    icon: '🚗',
    subcategories: [
      { name: 'Car Parts', description: 'Engine parts, brakes, and components', icon: '🔧' },
      { name: 'Car Accessories', description: 'Interior and exterior car accessories', icon: '🚗' },
      { name: 'Tires & Wheels', description: 'Tires, rims, and wheel accessories', icon: '🛞' },
      { name: 'Car Care', description: 'Cleaning products and maintenance supplies', icon: '🧽' },
      { name: 'Electronics', description: 'Car audio, GPS, and electronic accessories', icon: '📻' },
      { name: 'Tools & Equipment', description: 'Automotive tools and garage equipment', icon: '🔨' },
      { name: 'Motorcycle', description: 'Motorcycle parts and accessories', icon: '🏍️' },
      { name: 'RV & Camping', description: 'RV accessories and camping gear', icon: '🚐' },
      { name: 'Boat & Marine', description: 'Marine parts and boating accessories', icon: '⛵' },
      { name: 'Safety & Security', description: 'Car alarms, dash cams, and safety gear', icon: '🚨' }
    ]
  },
  {
    name: 'Food & Beverages',
    description: 'Food products, beverages, and culinary supplies',
    icon: '🍎',
    subcategories: [
      { name: 'Fresh Produce', description: 'Fruits, vegetables, and fresh items', icon: '🥕' },
      { name: 'Meat & Seafood', description: 'Fresh and frozen meat and seafood', icon: '🥩' },
      { name: 'Dairy & Eggs', description: 'Milk, cheese, yogurt, and eggs', icon: '🥛' },
      { name: 'Pantry Staples', description: 'Rice, pasta, canned goods, and essentials', icon: '🥫' },
      { name: 'Snacks & Candy', description: 'Chips, cookies, candy, and treats', icon: '🍪' },
      { name: 'Beverages', description: 'Soft drinks, juices, coffee, and tea', icon: '☕' },
      { name: 'Frozen Foods', description: 'Frozen meals, vegetables, and desserts', icon: '🧊' },
      { name: 'Bakery', description: 'Bread, pastries, and baked goods', icon: '🥖' },
      { name: 'International Foods', description: 'Ethnic and specialty food products', icon: '🌍' },
      { name: 'Organic & Health', description: 'Organic, gluten-free, and health foods', icon: '🌱' }
    ]
  },
  {
    name: 'Toys & Games',
    description: 'Toys, games, and entertainment for all ages',
    icon: '🧸',
    subcategories: [
      { name: 'Action Figures', description: 'Action figures and collectible toys', icon: '🦸' },
      { name: 'Dolls & Accessories', description: 'Dolls, dollhouses, and accessories', icon: '👸' },
      { name: 'Building Blocks', description: 'LEGO, building sets, and construction toys', icon: '🧱' },
      { name: 'Board Games', description: 'Family games, strategy games, and puzzles', icon: '🎲' },
      { name: 'Educational Toys', description: 'Learning toys and STEM products', icon: '🔬' },
      { name: 'Arts & Crafts', description: 'Art supplies, craft kits, and creativity tools', icon: '🎨' },
      { name: 'Outdoor Toys', description: 'Bikes, sports toys, and outdoor play equipment', icon: '🛴' },
      { name: 'Baby & Toddler', description: 'Toys for infants and toddlers', icon: '🧸' },
      { name: 'Remote Control', description: 'RC cars, drones, and remote-controlled toys', icon: '🚁' },
      { name: 'Collectibles', description: 'Trading cards, figurines, and collectibles', icon: '🃏' }
    ]
  },
  {
    name: 'Pet Supplies',
    description: 'Pet food, toys, and care products',
    icon: '🐕',
    subcategories: [
      { name: 'Dog Supplies', description: 'Dog food, toys, leashes, and accessories', icon: '🐕' },
      { name: 'Cat Supplies', description: 'Cat food, litter, toys, and accessories', icon: '🐱' },
      { name: 'Fish & Aquatic', description: 'Aquariums, fish food, and aquatic supplies', icon: '🐠' },
      { name: 'Bird Supplies', description: 'Bird cages, food, and accessories', icon: '🦜' },
      { name: 'Small Animals', description: 'Supplies for rabbits, hamsters, and small pets', icon: '🐹' },
      { name: 'Reptile & Exotic', description: 'Terrariums and exotic pet supplies', icon: '🦎' },
      { name: 'Pet Health', description: 'Vitamins, medications, and health products', icon: '💊' },
      { name: 'Pet Grooming', description: 'Grooming tools, shampoos, and care products', icon: '✂️' },
      { name: 'Pet Training', description: 'Training aids, treats, and behavior products', icon: '🎾' },
      { name: 'Pet Travel', description: 'Carriers, travel accessories, and safety gear', icon: '🧳' }
    ]
  },
  {
    name: 'Office & School',
    description: 'Office supplies, stationery, and educational materials',
    icon: '📝',
    subcategories: [
      { name: 'Writing Supplies', description: 'Pens, pencils, markers, and writing tools', icon: '✏️' },
      { name: 'Paper Products', description: 'Notebooks, paper, and printing supplies', icon: '📄' },
      { name: 'Desk Accessories', description: 'Organizers, staplers, and desk supplies', icon: '📎' },
      { name: 'Filing & Storage', description: 'Folders, binders, and storage solutions', icon: '🗂️' },
      { name: 'Art Supplies', description: 'Drawing, painting, and craft materials', icon: '🖍️' },
      { name: 'School Supplies', description: 'Backpacks, lunch boxes, and school essentials', icon: '🎒' },
      { name: 'Office Furniture', description: 'Desks, chairs, and office furniture', icon: '🪑' },
      { name: 'Technology', description: 'Calculators, printers, and office electronics', icon: '🖥️' },
      { name: 'Presentation', description: 'Whiteboards, projectors, and presentation tools', icon: '📊' },
      { name: 'Calendars & Planners', description: 'Calendars, planners, and scheduling tools', icon: '📅' }
    ]
  },
  {
    name: 'Travel & Luggage',
    description: 'Luggage, travel accessories, and vacation essentials',
    icon: '🧳',
    subcategories: [
      { name: 'Suitcases', description: 'Hard and soft shell suitcases', icon: '🧳' },
      { name: 'Backpacks', description: 'Travel and hiking backpacks', icon: '🎒' },
      { name: 'Travel Accessories', description: 'Travel pillows, organizers, and accessories', icon: '🧳' },
      { name: 'Carry-On Bags', description: 'Small luggage and carry-on items', icon: '👜' },
      { name: 'Travel Electronics', description: 'Adapters, chargers, and travel gadgets', icon: '🔌' },
      { name: 'Travel Comfort', description: 'Pillows, blankets, and comfort items', icon: '😴' },
      { name: 'Security', description: 'Locks, safes, and travel security items', icon: '🔒' },
      { name: 'Maps & Guides', description: 'Travel guides, maps, and reference materials', icon: '🗺️' },
      { name: 'Outdoor Gear', description: 'Camping and hiking travel equipment', icon: '⛺' },
      { name: 'Business Travel', description: 'Professional travel accessories and cases', icon: '💼' }
    ]
  }
];

let categoryCount = 0;
let subcategoryCount = 0;

// Create categories and subcategories
categoriesWithSubcategories.forEach((catData, catIndex) => {
  // Create category
  const categoryId = `CAT-${Date.now()}-${(categoryCount + 1).toString().padStart(3, '0')}`;
  const category = {
    id: categoryId,
    name: catData.name,
    description: catData.description,
    slug: catData.name.toLowerCase()
      .replace(/[^a-z0-9 -]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim(),
    icon: catData.icon,
    isActive: true,
    sortOrder: catIndex + 1,
    createdAt: new Date(Date.now() - Math.random() * 90 * 24 * 60 * 60 * 1000).toISOString(), // Random date within last 3 months
    updatedAt: new Date().toISOString(),
    meta: {
      revision: 0,
      created: Date.now(),
      version: 0
    },
    $loki: categoryCount + 1
  };
  
  categoriesCollection.data.push(category);
  categoryCount++;
  
  console.log(`✅ Category: ${catData.icon} ${catData.name} (${catData.subcategories.length} subcategories)`);
  
  // Create subcategories for this category
  catData.subcategories.forEach((subData, subIndex) => {
    const subcategoryId = `SUBCAT-${Date.now()}-${(subcategoryCount + 1).toString().padStart(3, '0')}`;
    const subcategory = {
      id: subcategoryId,
      name: subData.name,
      description: subData.description,
      slug: subData.name.toLowerCase()
        .replace(/[^a-z0-9 -]/g, '')
        .replace(/\s+/g, '-')
        .replace(/-+/g, '-')
        .trim(),
      categoryId: categoryId,
      icon: subData.icon,
      isActive: Math.random() > 0.05, // 95% active
      sortOrder: subIndex + 1,
      createdAt: new Date(Date.now() - Math.random() * 60 * 24 * 60 * 60 * 1000).toISOString(), // Random date within last 2 months
      updatedAt: new Date().toISOString(),
      meta: {
        revision: 0,
        created: Date.now(),
        version: 0
      },
      $loki: subcategoryCount + 1
    };
    
    subcategoriesCollection.data.push(subcategory);
    subcategoryCount++;
    
    console.log(`   📂 ${subData.icon} ${subData.name}`);
  });
  
  console.log(''); // Empty line for readability
});

// Update collection metadata
categoriesCollection.maxId = categoryCount;
categoriesCollection.dirty = true;
subcategoriesCollection.maxId = subcategoryCount;
subcategoriesCollection.dirty = true;

// Write the updated database back to file
try {
  fs.writeFileSync(dbPath, JSON.stringify(dbData, null, 2));
  console.log('🎉 SUCCESS! Database updated with comprehensive categories and subcategories!');
  console.log('');
  console.log('📊 Summary:');
  console.log(`   📁 Categories created: ${categoryCount}`);
  console.log(`   📂 Subcategories created: ${subcategoryCount}`);
  console.log(`   🎯 Total items: ${categoryCount + subcategoryCount}`);
  console.log('');
  
  // Show some stats
  const activeCategories = categoriesCollection.data.filter(c => c.isActive).length;
  const activeSubcategories = subcategoriesCollection.data.filter(sc => sc.isActive).length;
  
  console.log('📈 Status:');
  console.log(`   ✅ Active categories: ${activeCategories}/${categoryCount}`);
  console.log(`   ✅ Active subcategories: ${activeSubcategories}/${subcategoryCount}`);
  console.log('');
  console.log('🚀 Ready to use! Visit your dashboard to see all the new categories and subcategories.');
  
} catch (error) {
  console.error('❌ Error writing to database:', error);
  process.exit(1);
}