const loki = require('lokijs');

function copyDatabase() {
  // Load source database (users.db)
  const sourceDb = new loki('users.db');
  
  sourceDb.loadDatabase({}, () => {
    console.log('📂 Source database loaded');
    
    // Load target database (ecommerce.db)
    const targetDb = new loki('ecommerce.db');
    
    targetDb.loadDatabase({}, () => {
      console.log('📂 Target database loaded');
      
      // Get collections from source
      const sourceCategories = sourceDb.getCollection('categories');
      const sourceSubcategories = sourceDb.getCollection('subcategories');
      const sourceProducts = sourceDb.getCollection('products');
      
      // Get or create collections in target
      let targetCategories = targetDb.getCollection('categories');
      let targetSubcategories = targetDb.getCollection('subcategories');
      let targetProducts = targetDb.getCollection('products');
      
      if (!targetCategories) {
        targetCategories = targetDb.addCollection('categories');
      }
      if (!targetSubcategories) {
        targetSubcategories = targetDb.addCollection('subcategories');
      }
      if (!targetProducts) {
        targetProducts = targetDb.addCollection('products');
      }
      
      // Clear existing data
      targetCategories.clear();
      targetSubcategories.clear();
      targetProducts.clear();
      
      // Copy categories
      if (sourceCategories) {
        const categories = sourceCategories.find({});
        categories.forEach(cat => {
          // Remove LokiJS internal properties
          const cleanCat = { ...cat };
          delete cleanCat.$loki;
          delete cleanCat.meta;
          targetCategories.insert(cleanCat);
        });
        console.log(`✅ Copied ${categories.length} categories`);
      }
      
      // Copy subcategories
      if (sourceSubcategories) {
        const subcategories = sourceSubcategories.find({});
        subcategories.forEach(sub => {
          const cleanSub = { ...sub };
          delete cleanSub.$loki;
          delete cleanSub.meta;
          targetSubcategories.insert(cleanSub);
        });
        console.log(`✅ Copied ${subcategories.length} subcategories`);
      }
      
      // Copy products
      if (sourceProducts) {
        const products = sourceProducts.find({});
        products.forEach(prod => {
          const cleanProd = { ...prod };
          delete cleanProd.$loki;
          delete cleanProd.meta;
          targetProducts.insert(cleanProd);
        });
        console.log(`✅ Copied ${products.length} products`);
      }
      
      // Save target database
      targetDb.saveDatabase(() => {
        console.log('💾 Target database saved');
        
        // Verify the copy
        const finalCategories = targetCategories.count();
        const finalSubcategories = targetSubcategories.count();
        const finalProducts = targetProducts.count();
        const simpleProducts = targetProducts.count({ type: 'simple' });
        const variableProducts = targetProducts.count({ type: 'variable' });
        
        console.log('\\n🎉 DATABASE COPY COMPLETED!');
        console.log('📊 Final Statistics:');
        console.log(`   📁 Categories: ${finalCategories}`);
        console.log(`   📂 Subcategories: ${finalSubcategories}`);
        console.log(`   📦 Total Products: ${finalProducts}`);
        console.log(`   🔧 Simple Products: ${simpleProducts}`);
        console.log(`   🎛️  Variable Products: ${variableProducts}`);
        
        console.log('\\n✅ ecommerce.db now contains all the data!');
      });
    });
  });
}

copyDatabase();