const fs = require('fs');
const path = require('path');

// Convert existing simple products to variable products
function convertToVariableProducts() {
  const dbPath = path.join(__dirname, 'users.db');
  
  if (!fs.existsSync(dbPath)) {
    console.error('Database file not found.');
    return;
  }

  const db = JSON.parse(fs.readFileSync(dbPath, 'utf8'));
  
  // Get existing data
  const categories = db.collections[2].data; // categories
  const products = db.collections[4].data; // products
  
  console.log(`Found ${categories.length} categories and ${products.length} products`);
  
  // Product attributes for variable products
  const productAttributes = {
    'electronics': [
      { name: 'Storage', values: ['64GB', '128GB', '256GB', '512GB'] },
      { name: 'Color', values: ['Black', 'White', 'Silver', 'Gold'] }
    ],
    'clothing': [
      { name: 'Size', values: ['XS', 'S', 'M', 'L', 'XL'] },
      { name: 'Color', values: ['Black', 'White', 'Red', 'Blue', 'Green'] }
    ],
    'sports': [
      { name: 'Size', values: ['S', 'M', 'L', 'XL'] },
      { name: 'Color', values: ['Black', 'Red', 'Blue', 'White'] }
    ],
    'home-garden': [
      { name: 'Size', values: ['Small', 'Medium', 'Large'] },
      { name: 'Material', values: ['Wood', 'Metal', 'Plastic'] }
    ]
  };

  // Categories that should have variable products
  const variableCategories = ['electronics', 'clothing', 'sports', 'home-garden'];
  
  let convertedCount = 0;
  
  // Convert 30% of products in variable categories to variable products
  products.forEach((product, index) => {
    // Skip if already processed or if random chance doesn't hit
    if (product.type || Math.random() > 0.3) {
      // Make it a simple product if not already set
      if (!product.type) {
        product.type = 'simple';
      }
      return;
    }
    
    // Find the category for this product
    const category = categories.find(cat => cat.id === product.categoryId || cat.$loki === parseInt(product.categoryId));
    if (!category || !variableCategories.includes(category.slug)) {
      product.type = 'simple';
      return;
    }
    
    // Convert to variable product
    const categoryAttrs = productAttributes[category.slug] || productAttributes.clothing;
    const selectedAttrs = categoryAttrs.slice(0, Math.random() > 0.6 ? 2 : 1);
    
    // Generate variants for this product
    const variants = [];
    const generateVariants = (attrs, index = 0, currentCombination = {}) => {
      if (index === attrs.length) {
        const variantPrice = product.price + (Math.floor(Math.random() * 100) - 50); // ±$50 variation
        const variantId = `${product.id}-var-${variants.length + 1}`;
        
        variants.push({
          id: variantId,
          sku: `SKU-${variantId.toUpperCase()}`,
          price: Math.max(variantPrice, 10),
          comparePrice: product.comparePrice ? (variantPrice + Math.floor(Math.random() * 50) + 20) : undefined,
          stockQuantity: Math.floor(Math.random() * 30) + 5,
          attributes: { ...currentCombination },
          imageUrl: product.imageUrl,
          isActive: true
        });
        return;
      }

      const attr = attrs[index];
      // Limit combinations to avoid too many variants
      const valuesToUse = attr.values.slice(0, Math.min(4, attr.values.length));
      valuesToUse.forEach(value => {
        generateVariants(attrs, index + 1, { ...currentCombination, [attr.name]: value });
      });
    };

    generateVariants(selectedAttrs);

    // Calculate price range
    const prices = variants.map(v => v.price);
    const priceRange = {
      min: Math.min(...prices),
      max: Math.max(...prices)
    };

    // Convert the product to variable
    product.type = 'variable';
    product.attributes = selectedAttrs;
    product.variants = variants;
    product.priceRange = priceRange;
    product.images = [product.imageUrl, product.imageUrl, product.imageUrl]; // Use same image for now
    
    // Remove simple product properties that are now in variants
    delete product.price;
    delete product.comparePrice;
    delete product.stockQuantity;
    delete product.sku;
    delete product.imageUrl;
    
    convertedCount++;
  });
  
  // Write updated database
  fs.writeFileSync(dbPath, JSON.stringify(db, null, 2));
  
  const totalProducts = products.length;
  const simpleProducts = products.filter(p => p.type === 'simple').length;
  const variableProducts = products.filter(p => p.type === 'variable').length;
  const totalVariants = products
    .filter(p => p.type === 'variable')
    .reduce((sum, p) => sum + (p.variants ? p.variants.length : 0), 0);

  console.log('✅ Product type conversion completed!');
  console.log('📊 Product Statistics:');
  console.log(`   📦 Total Products: ${totalProducts}`);
  console.log(`   🔧 Simple Products: ${simpleProducts} (${Math.round((simpleProducts / totalProducts) * 100)}%)`);
  console.log(`   🎛️  Variable Products: ${variableProducts} (${Math.round((variableProducts / totalProducts) * 100)}%)`);
  console.log(`   📋 Total Variants: ${totalVariants}`);
  console.log(`   🔄 Products Converted: ${convertedCount}`);
  
  if (variableProducts > 0) {
    const avgVariantsPerProduct = Math.round(totalVariants / variableProducts);
    console.log(`   📈 Average Variants per Variable Product: ${avgVariantsPerProduct}`);
  }
}

convertToVariableProducts();