const loki = require('lokijs');

const db = new loki('users.db');
db.loadDatabase({}, () => {
  console.log('🔍 COMPREHENSIVE CATEGORY & SUBCATEGORY VERIFICATION');
  console.log('===================================================\n');
  
  const categories = db.getCollection('categories');
  const subcategories = db.getCollection('subcategories');
  
  if (!categories || !subcategories) {
    console.error('❌ Collections not found');
    return;
  }
  
  const allCategories = categories.find({});
  const allSubcategories = subcategories.find({});
  
  console.log(`📊 OVERVIEW:`);
  console.log(`Total Categories: ${allCategories.length}`);
  console.log(`Total Subcategories: ${allSubcategories.length}`);
  console.log(`Active Categories: ${allCategories.filter(c => c.isActive).length}`);
  console.log(`Active Subcategories: ${allSubcategories.filter(s => s.isActive).length}\n`);
  
  console.log(`📋 DETAILED BREAKDOWN:`);
  console.log('=====================');
  
  let totalSubcategoryCount = 0;
  let categoriesWithoutEnoughSubs = 0;
  let minSubcategories = Infinity;
  let maxSubcategories = 0;
  
  allCategories.forEach((cat, index) => {
    const categorySubcategories = allSubcategories.filter(sub => sub.categoryId === cat.id);
    const count = categorySubcategories.length;
    
    totalSubcategoryCount += count;
    
    if (count < minSubcategories) minSubcategories = count;
    if (count > maxSubcategories) maxSubcategories = count;
    
    const status = count >= 5 ? '✅' : count >= 3 ? '⚠️ ' : '❌';
    
    if (count < 5) categoriesWithoutEnoughSubs++;
    
    console.log(`${status} ${cat.name.padEnd(25)} | ${count.toString().padStart(2)} subcategories | ID: ${cat.id}`);
    
    // Show first few subcategories for verification
    if (count > 0 && index < 5) {
      categorySubcategories.slice(0, 3).forEach(sub => {
        console.log(`    └─ ${sub.name} (${sub.slug})`);
      });
      if (count > 3) {
        console.log(`    └─ ... and ${count - 3} more`);
      }
    }
  });
  
  console.log('\n📈 STATISTICS:');
  console.log('==============');
  console.log(`Minimum subcategories per category: ${minSubcategories}`);
  console.log(`Maximum subcategories per category: ${maxSubcategories}`);
  console.log(`Average subcategories per category: ${(totalSubcategoryCount / allCategories.length).toFixed(1)}`);
  console.log(`Categories with <5 subcategories: ${categoriesWithoutEnoughSubs}`);
  
  console.log('\n🎯 VALIDATION RESULTS:');
  console.log('======================');
  
  if (categoriesWithoutEnoughSubs === 0) {
    console.log('✅ PERFECT: All categories have at least 5 subcategories!');
  } else {
    console.log(`❌ WARNING: ${categoriesWithoutEnoughSubs} categories need more subcategories`);
  }
  
  if (allCategories.length >= 35) {
    console.log('✅ EXCELLENT: Comprehensive category coverage (35+ categories)');
  } else {
    console.log(`⚠️  Category count could be improved (current: ${allCategories.length})`);
  }
  
  if (allSubcategories.length >= 200) {
    console.log('✅ OUTSTANDING: Rich subcategory diversity (200+ subcategories)');
  } else {
    console.log(`⚠️  Subcategory count could be improved (current: ${allSubcategories.length})`);
  }
  
  console.log('\n🚀 SYSTEM STATUS: READY FOR E-COMMERCE! 🛍️');
});