const loki = require('lokijs');
const http = require('http');

console.log('🔍 COMPREHENSIVE DATA AVAILABILITY CHECK');
console.log('========================================\n');

// First check database directly
const db = new loki('users.db');

db.loadDatabase({}, () => {
  console.log('📊 DATABASE VERIFICATION:');
  console.log('=========================');
  
  const categories = db.getCollection('categories');
  const subcategories = db.getCollection('subcategories');
  const products = db.getCollection('products');
  
  if (!categories || !subcategories || !products) {
    console.error('❌ Collections not found in database');
    return;
  }
  
  const allCategories = categories.find({});
  const allSubcategories = subcategories.find({});
  const allProducts = products.find({});
  
  console.log(`✅ Categories in database: ${allCategories.length}/35`);
  console.log(`✅ Subcategories in database: ${allSubcategories.length}/244`);
  console.log(`✅ Products in database: ${allProducts.length}/1000+`);
  
  // Check active status
  const activeCategories = allCategories.filter(c => c.isActive);
  const activeSubcategories = allSubcategories.filter(s => s.isActive);
  const activeProducts = allProducts.filter(p => p.isActive);
  
  console.log(`✅ Active categories: ${activeCategories.length}`);
  console.log(`✅ Active subcategories: ${activeSubcategories.length}`);
  console.log(`✅ Active products: ${activeProducts.length}`);
  
  // Check data integrity
  const categoriesWithSubcategories = new Set();
  const subcategoriesWithProducts = new Set();
  
  allSubcategories.forEach(sub => {
    if (sub.categoryId) categoriesWithSubcategories.add(sub.categoryId);
  });
  
  allProducts.forEach(product => {
    if (product.subcategoryId) subcategoriesWithProducts.add(product.subcategoryId);
  });
  
  console.log(`✅ Categories with subcategories: ${categoriesWithSubcategories.size}/${allCategories.length}`);
  console.log(`✅ Subcategories with products: ${subcategoriesWithProducts.size}/${allSubcategories.length}`);
  
  // Sample some data
  console.log('\n📋 SAMPLE DATA VERIFICATION:');
  console.log('============================');
  
  if (allCategories.length > 0) {
    const sampleCategory = allCategories[0];
    console.log(`Sample Category: ${sampleCategory.name} (ID: ${sampleCategory.id}, Slug: ${sampleCategory.slug})`);
  }
  
  if (allSubcategories.length > 0) {
    const sampleSubcategory = allSubcategories[0];
    console.log(`Sample Subcategory: ${sampleSubcategory.name} (ID: ${sampleSubcategory.id}, Category: ${sampleSubcategory.categoryId})`);
  }
  
  if (allProducts.length > 0) {
    const sampleProduct = allProducts[0];
    console.log(`Sample Product: ${sampleProduct.name} (Price: $${sampleProduct.price}, Category: ${sampleProduct.categoryId})`);
  }
  
  // Check for any missing relationships
  const orphanedSubcategories = allSubcategories.filter(sub => 
    !allCategories.find(cat => cat.id === sub.categoryId)
  );
  
  const orphanedProducts = allProducts.filter(product => 
    !allSubcategories.find(sub => sub.id === product.subcategoryId) ||
    !allCategories.find(cat => cat.id === product.categoryId)
  );
  
  if (orphanedSubcategories.length > 0) {
    console.log(`⚠️  Orphaned subcategories: ${orphanedSubcategories.length}`);
  } else {
    console.log('✅ No orphaned subcategories');
  }
  
  if (orphanedProducts.length > 0) {
    console.log(`⚠️  Orphaned products: ${orphanedProducts.length}`);
  } else {
    console.log('✅ No orphaned products');
  }
  
  console.log('\n🌐 API ENDPOINT TESTING:');
  console.log('========================');
  
  // Test API endpoints
  const testEndpoint = (path, expectedName) => {
    return new Promise((resolve) => {
      const options = {
        hostname: 'localhost',
        port: 3000,
        path: path,
        method: 'GET',
        timeout: 5000
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        
        res.on('data', (chunk) => {
          data += chunk;
        });
        
        res.on('end', () => {
          try {
            const jsonData = JSON.parse(data);
            const count = jsonData.categories ? jsonData.categories.length : 
                         jsonData.subcategories ? jsonData.subcategories.length :
                         jsonData.products ? jsonData.products.length : 0;
            console.log(`✅ ${expectedName} API: ${count} items (Status: ${res.statusCode})`);
            resolve(count);
          } catch (error) {
            console.log(`❌ ${expectedName} API: Parse error (Status: ${res.statusCode})`);
            resolve(0);
          }
        });
      });
      
      req.on('error', (error) => {
        console.log(`❌ ${expectedName} API: Connection failed - ${error.message}`);
        resolve(0);
      });
      
      req.on('timeout', () => {
        console.log(`❌ ${expectedName} API: Timeout`);
        req.destroy();
        resolve(0);
      });
      
      req.end();
    });
  };
  
  // Test all endpoints
  Promise.all([
    testEndpoint('/api/categories', 'Categories'),
    testEndpoint('/api/subcategories', 'Subcategories'),  
    testEndpoint('/api/products', 'Products')
  ]).then(([categoriesCount, subcategoriesCount, productsCount]) => {
    console.log('\n📊 FINAL SUMMARY:');
    console.log('=================');
    
    const dbCategoriesOK = allCategories.length >= 35;
    const dbSubcategoriesOK = allSubcategories.length >= 244;
    const dbProductsOK = allProducts.length >= 1000;
    
    const apiCategoriesOK = categoriesCount >= 35;
    const apiSubcategoriesOK = subcategoriesCount >= 244;
    const apiProductsOK = productsCount >= 1000;
    
    console.log(`Database Categories: ${dbCategoriesOK ? '✅' : '❌'} ${allCategories.length}/35`);
    console.log(`Database Subcategories: ${dbSubcategoriesOK ? '✅' : '❌'} ${allSubcategories.length}/244`);
    console.log(`Database Products: ${dbProductsOK ? '✅' : '❌'} ${allProducts.length}/1000+`);
    
    console.log(`API Categories: ${apiCategoriesOK ? '✅' : '❌'} ${categoriesCount}/35`);
    console.log(`API Subcategories: ${apiSubcategoriesOK ? '✅' : '❌'} ${subcategoriesCount}/244`);
    console.log(`API Products: ${apiProductsOK ? '✅' : '❌'} ${productsCount}/1000+`);
    
    const allSystemsGo = dbCategoriesOK && dbSubcategoriesOK && dbProductsOK && 
                        apiCategoriesOK && apiSubcategoriesOK && apiProductsOK;
    
    if (allSystemsGo) {
      console.log('\n🎉 ALL SYSTEMS GO! ✨');
      console.log('==================');
      console.log('✅ 35 Categories available');
      console.log('✅ 244 Subcategories available');
      console.log('✅ 1000+ Products available');
      console.log('✅ Database operational');
      console.log('✅ APIs responding correctly');
      console.log('\n🚀 Your e-commerce platform is fully operational! 🛍️');
    } else {
      console.log('\n⚠️  SYSTEM ISSUES DETECTED');
      console.log('=========================');
      if (!dbCategoriesOK || !apiCategoriesOK) console.log('❌ Categories issue');
      if (!dbSubcategoriesOK || !apiSubcategoriesOK) console.log('❌ Subcategories issue');
      if (!dbProductsOK || !apiProductsOK) console.log('❌ Products issue');
    }
  }).catch(error => {
    console.error('❌ API testing failed:', error);
    console.log('\n📊 DATABASE-ONLY SUMMARY:');
    console.log('=========================');
    console.log(`✅ Categories: ${allCategories.length}/35`);
    console.log(`✅ Subcategories: ${allSubcategories.length}/244`);
    console.log(`✅ Products: ${allProducts.length}/1000+`);
  });
});