const loki = require('lokijs');

// Initialize database
const db = new loki('users.db');

db.loadDatabase({}, () => {
  console.log('🔧 COMPLETE SYSTEM RESTORATION');
  console.log('==============================\n');
  
  // Clear and recreate all collections
  const collections = ['categories', 'subcategories', 'products'];
  collections.forEach(collectionName => {
    const collection = db.getCollection(collectionName);
    if (collection) {
      db.removeCollection(collectionName);
    }
  });
  
  // Create fresh collections
  const categories = db.addCollection('categories');
  const subcategories = db.addCollection('subcategories');
  const products = db.addCollection('products');
  
  console.log('✅ Collections recreated');
  
  // Add all 35 categories
  const categoryData = [
    { id: '1', name: 'Electronics', description: 'Electronic devices and gadgets', icon: '📱', slug: 'electronics' },
    { id: '2', name: 'Fashion', description: 'Clothing and fashion accessories', icon: '👕', slug: 'fashion' },
    { id: '3', name: 'Home & Garden', description: 'Home improvement and garden supplies', icon: '🏠', slug: 'home-garden' },
    { id: '4', name: 'Sports', description: 'Sports equipment and fitness gear', icon: '⚽', slug: 'sports' },
    { id: '5', name: 'Documents / Files', description: 'Digital and physical document management', icon: '📄', slug: 'documents-files' },
    { id: '6', name: 'Shopping / Retail', description: 'Shopping tools and retail services', icon: '🛒', slug: 'shopping-retail' },
    { id: '7', name: 'Clothing', description: 'All types of clothing and apparel', icon: '👔', slug: 'clothing' },
    { id: '8', name: 'Footwear', description: 'Shoes and footwear for all occasions', icon: '👟', slug: 'footwear' },
    { id: '9', name: 'Mobile Devices', description: 'Smartphones, tablets and mobile accessories', icon: '📱', slug: 'mobile-devices' },
    { id: '10', name: 'Computers / Tech', description: 'Computers and technology products', icon: '💻', slug: 'computers-tech' },
    { id: '11', name: 'Home / Real Estate', description: 'Home decor and real estate', icon: '🏘️', slug: 'home-real-estate' },
    { id: '12', name: 'Gaming', description: 'Video games and gaming equipment', icon: '🎮', slug: 'gaming' },
    { id: '13', name: 'Books / Education', description: 'Books and educational materials', icon: '📚', slug: 'books-education' },
    { id: '14', name: 'Music', description: 'Musical instruments and audio equipment', icon: '🎵', slug: 'music' },
    { id: '15', name: 'Food', description: 'Food products and groceries', icon: '🍎', slug: 'food' },
    { id: '16', name: 'Coffee / Beverages', description: 'Coffee, tea and beverages', icon: '☕', slug: 'coffee-beverages' },
    { id: '17', name: 'Beauty & Personal Care', description: 'Beauty products and personal care items', icon: '💄', slug: 'beauty-personal-care' },
    { id: '18', name: 'Health & Wellness', description: 'Health products and wellness items', icon: '🏥', slug: 'health-wellness' },
    { id: '19', name: 'Toys & Kids', description: 'Toys and children\'s products', icon: '🧸', slug: 'toys-kids' },
    { id: '20', name: 'Sports & Outdoors', description: 'Sports equipment and outdoor gear', icon: '🏃', slug: 'sports-outdoors' },
    { id: '21', name: 'Automotive', description: 'Car parts and automotive accessories', icon: '🚗', slug: 'automotive' },
    { id: '22', name: 'Jewelry & Accessories', description: 'Jewelry and fashion accessories', icon: '💎', slug: 'jewelry-accessories' },
    { id: '23', name: 'Pet Supplies', description: 'Pet food and animal supplies', icon: '🐕', slug: 'pet-supplies' },
    { id: '24', name: 'Furniture', description: 'Home and office furniture', icon: '🛋️', slug: 'furniture' },
    { id: '25', name: 'Appliances', description: 'Home and kitchen appliances', icon: '🔌', slug: 'appliances' },
    { id: '26', name: 'Groceries', description: 'Food and household essentials', icon: '🛍️', slug: 'groceries' },
    { id: '27', name: 'Office Supplies', description: 'Office and business supplies', icon: '📎', slug: 'office-supplies' },
    { id: '28', name: 'Travel & Luggage', description: 'Travel accessories and luggage', icon: '🧳', slug: 'travel-luggage' },
    { id: '29', name: 'Art & Crafts', description: 'Art supplies and craft materials', icon: '🎨', slug: 'art-crafts' },
    { id: '30', name: 'Seasonal & Gifts', description: 'Seasonal items and gift products', icon: '🎁', slug: 'seasonal-gifts' },
    { id: '31', name: 'Featured / Favorites', description: 'Featured and favorite products', icon: '⭐', slug: 'featured-favorites' },
    { id: '32', name: 'Ideas / Innovation', description: 'Innovative products and ideas', icon: '💡', slug: 'ideas-innovation' },
    { id: '33', name: 'Productivity / Goals', description: 'Productivity tools and goal-setting', icon: '🎯', slug: 'productivity-goals' },
    { id: '34', name: 'Tools / Maintenance', description: 'Tools and maintenance supplies', icon: '🔧', slug: 'tools-maintenance' },
    { id: '35', name: 'Finance / Subscriptions', description: 'Financial services and subscriptions', icon: '💳', slug: 'finance-subscriptions' }
  ];
  
  // Add categories
  categoryData.forEach(cat => {
    categories.insert({
      ...cat,
      isActive: true,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    });
  });
  
  console.log('✅ Added 35 categories');
  
  // Add subcategories with proper relationships
  const subcategoryData = [
    // Electronics subcategories
    { name: 'Audio & Headphones', categoryId: '1', slug: 'audio-headphones', icon: '🎧' },
    { name: 'Cameras', categoryId: '1', slug: 'cameras', icon: '📷' },
    { name: 'Smart Watches', categoryId: '1', slug: 'smart-watches', icon: '⌚' },
    { name: 'TV & Home Theater', categoryId: '1', slug: 'tv-home-theater', icon: '📺' },
    { name: 'Computer Accessories', categoryId: '1', slug: 'computer-accessories', icon: '⌨️' },
    { name: 'Cables & Chargers', categoryId: '1', slug: 'cables-chargers', icon: '🔌' },
    
    // Fashion subcategories
    { name: 'Shoes', categoryId: '2', slug: 'shoes', icon: '👟' },
    { name: 'Accessories', categoryId: '2', slug: 'fashion-accessories', icon: '👜' },
    { name: 'Formal Wear', categoryId: '2', slug: 'formal-wear', icon: '🤵' },
    { name: 'Swimwear', categoryId: '2', slug: 'swimwear', icon: '👙' },
    { name: 'Lingerie & Underwear', categoryId: '2', slug: 'lingerie-underwear', icon: '👙' },
    
    // Home & Garden subcategories
    { name: 'Furniture', categoryId: '3', slug: 'home-furniture', icon: '🛋️' },
    { name: 'Bedding & Bath', categoryId: '3', slug: 'bedding-bath', icon: '🛏️' },
    { name: 'Storage & Organization', categoryId: '3', slug: 'storage-organization', icon: '📦' },
    { name: 'Tools & Hardware', categoryId: '3', slug: 'tools-hardware', icon: '🔨' },
    { name: 'Cleaning Supplies', categoryId: '3', slug: 'cleaning-supplies', icon: '🧽' },
    
    // Sports subcategories
    { name: 'Athletic Wear', categoryId: '4', slug: 'athletic-wear', icon: '👕' },
    { name: 'Running & Cardio', categoryId: '4', slug: 'running-cardio', icon: '🏃' },
    { name: 'Strength Training', categoryId: '4', slug: 'strength-training', icon: '🏋️' },
    { name: 'Yoga & Pilates', categoryId: '4', slug: 'yoga-pilates', icon: '🧘' },
    { name: 'Golf', categoryId: '4', slug: 'golf', icon: '⛳' },
    { name: 'Tennis & Racquet Sports', categoryId: '4', slug: 'tennis-racquet-sports', icon: '🎾' }
  ];
  
  // Add 5+ subcategories for remaining categories
  for (let i = 5; i <= 35; i++) {
    const categoryId = i.toString();
    const category = categoryData.find(c => c.id === categoryId);
    
    for (let j = 1; j <= 6; j++) {
      subcategoryData.push({
        name: `${category.name} Subcategory ${j}`,
        categoryId: categoryId,
        slug: `${category.slug}-subcategory-${j}`,
        icon: category.icon
      });
    }
  }
  
  // Insert all subcategories
  subcategoryData.forEach((sub, index) => {
    subcategories.insert({
      id: (index + 1).toString(),
      ...sub,
      description: `High-quality ${sub.name.toLowerCase()} products and accessories`,
      isActive: true,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    });
  });
  
  console.log(`✅ Added ${subcategoryData.length} subcategories`);
  
  // Generate 1000+ products
  const productTemplates = [
    'Premium Product', 'Professional Item', 'Quality Tool', 'Essential Equipment', 'Popular Choice',
    'Best Seller', 'Customer Favorite', 'Recommended Item', 'Signature Product', 'Top-Rated Product'
  ];
  
  const brands = ['ProMax', 'TechCore', 'UltraFit', 'SwiftPro', 'PrimeLine', 'EliteCore'];
  const adjectives = ['Premium', 'Professional', 'Advanced', 'Ultra', 'Super', 'Deluxe'];
  
  function createSlug(name) {
    return name.toLowerCase()
      .replace(/[^a-z0-9 -]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim('-');
  }
  
  const allSubcategories = subcategories.find({});
  let productId = 1;
  let totalProducts = 0;
  
  // Generate 4-5 products per subcategory
  allSubcategories.forEach(subcategory => {
    const category = categories.findOne({ id: subcategory.categoryId });
    if (!category) return;
    
    const productsPerSub = Math.floor(Math.random() * 2) + 4; // 4-5 products
    
    for (let i = 0; i < productsPerSub && totalProducts < 1200; i++) {
      const template = productTemplates[Math.floor(Math.random() * productTemplates.length)];
      const brand = brands[Math.floor(Math.random() * brands.length)];
      const adjective = adjectives[Math.floor(Math.random() * adjectives.length)];
      
      const productName = `${brand} ${adjective} ${template}`;
      const basePrice = Math.round((Math.random() * 300 + 20) * 100) / 100;
      const comparePrice = Math.random() > 0.7 ? Math.round(basePrice * 1.3 * 100) / 100 : null;
      
      const product = {
        id: productId,
        name: productName,
        description: `High-quality ${productName.toLowerCase()} perfect for ${subcategory.name.toLowerCase()}. Features premium materials and excellent craftsmanship.`,
        slug: createSlug(productName) + '-' + Math.random().toString(36).substr(2, 4),
        price: basePrice,
        comparePrice: comparePrice,
        categoryId: category.id,
        subcategoryId: subcategory.id,
        categoryName: category.name,
        subcategoryName: subcategory.name,
        sku: `${category.name.substr(0, 3).toUpperCase()}-${String(productId).padStart(4, '0')}`,
        barcode: `${Math.floor(Math.random() * 9000000000000) + 1000000000000}`,
        stockQuantity: Math.floor(Math.random() * 100) + 10,
        lowStockAlert: 5,
        weight: Math.round((Math.random() * 5 + 0.1) * 100) / 100,
        dimensions: `${Math.floor(Math.random() * 30 + 5)} x ${Math.floor(Math.random() * 30 + 5)} x ${Math.floor(Math.random() * 20 + 2)} cm`,
        isActive: true,
        isFeatured: Math.random() > 0.8, // 20% chance of being featured
        tags: [
          subcategory.name.toLowerCase().replace(/\s+/g, '-'),
          category.name.toLowerCase().replace(/\s+/g, '-'),
          'premium', 'quality'
        ],
        imageUrl: `/images/products/${createSlug(template)}.jpg`,
        metaTitle: `${productName} - ${category.name}`,
        metaDescription: `Shop ${productName} in our ${subcategory.name} collection. High quality ${category.name.toLowerCase()} products.`,
        createdAt: new Date(Date.now() - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString(),
        updatedAt: new Date().toISOString()
      };
      
      products.insert(product);
      productId++;
      totalProducts++;
      
      if (totalProducts % 100 === 0) {
        console.log(`✅ Generated ${totalProducts} products...`);
      }
    }
  });
  
  // Save the database
  db.saveDatabase((err) => {
    if (err) {
      console.error('❌ Error saving database:', err);
    } else {
      console.log(`\n🎉 SYSTEM RESTORATION COMPLETE!`);
      console.log('================================');
      console.log(`✅ Categories: ${categories.count()}`);
      console.log(`✅ Subcategories: ${subcategories.count()}`);
      console.log(`✅ Products: ${products.count()}`);
      console.log(`✅ Featured products: ${products.find({ isFeatured: true }).length}`);
      
      // Verify relationships
      const categoriesWithProducts = new Set();
      const subcategoriesWithProducts = new Set();
      
      products.find({}).forEach(product => {
        if (product.categoryId) categoriesWithProducts.add(product.categoryId);
        if (product.subcategoryId) subcategoriesWithProducts.add(product.subcategoryId);
      });
      
      console.log(`✅ Categories with products: ${categoriesWithProducts.size}/${categories.count()}`);
      console.log(`✅ Subcategories with products: ${subcategoriesWithProducts.size}/${subcategories.count()}`);
      
      console.log('\n🚀 E-COMMERCE SYSTEM FULLY OPERATIONAL! 🛍️✨');
    }
  });
});