const loki = require('lokijs');

// Initialize database
const db = new loki('users.db');

db.loadDatabase({}, function() {
    console.log('=== COMPLETE CATEGORIES & SUBCATEGORIES OVERVIEW ===\n');
    
    // Get collections
    const categoriesCollection = db.getCollection('categories');
    const subcategoriesCollection = db.getCollection('subcategories');
    
    if (!categoriesCollection || !subcategoriesCollection) {
        console.log('❌ Collections not found');
        return;
    }
    
    // Get all categories sorted by name
    const categories = categoriesCollection.chain().simplesort('name').data();
    const subcategories = subcategoriesCollection.find({});
    
    console.log(`📊 SUMMARY:`);
    console.log(`   Total Categories: ${categories.length}`);
    console.log(`   Total Subcategories: ${subcategories.length}`);
    console.log(`   Database Status: ✅ Active\n`);
    
    console.log('📂 COMPLETE CATEGORIES & SUBCATEGORIES LIST:\n');
    
    // Display each category with its subcategories
    categories.forEach((category, index) => {
        const categorySubcategories = subcategories.filter(sub => sub.categoryId === category.$loki);
        
        console.log(`${index + 1}. 📂 ${category.name} (${category.slug})`);
        console.log(`   🔗 ID: ${category.$loki}`);
        console.log(`   📝 Description: ${category.description || 'No description'}`);
        console.log(`   🎯 Status: ${category.isActive ? '✅ Active' : '❌ Inactive'}`);
        console.log(`   📅 Created: ${category.createdAt ? new Date(category.createdAt).toLocaleDateString() : 'N/A'}`);
        console.log(`   📊 Subcategories: ${categorySubcategories.length}`);
        
        if (categorySubcategories.length > 0) {
            console.log(`   └── Subcategories:`);
            categorySubcategories.forEach((subcat, subIndex) => {
                console.log(`       ${subIndex + 1}. ${subcat.name} (${subcat.slug})`);
                console.log(`          📝 ${subcat.description || 'No description'}`);
                console.log(`          🎯 ${subcat.isActive ? '✅ Active' : '❌ Inactive'}`);
            });
        } else {
            console.log(`   └── ⚠️ No subcategories`);
        }
        console.log(''); // Empty line for readability
    });
    
    // Additional statistics
    console.log('📈 DETAILED STATISTICS:');
    
    // Categories with most subcategories
    const categoryStats = categories.map(cat => ({
        name: cat.name,
        count: subcategories.filter(sub => sub.categoryId === cat.$loki).length
    })).sort((a, b) => b.count - a.count);
    
    console.log('\n🏆 Top 10 Categories by Subcategory Count:');
    categoryStats.slice(0, 10).forEach((stat, index) => {
        console.log(`   ${index + 1}. ${stat.name}: ${stat.count} subcategories`);
    });
    
    // Check for any issues
    console.log('\n🔍 DATA INTEGRITY CHECK:');
    
    // Categories without subcategories
    const categoriesWithoutSubs = categoryStats.filter(stat => stat.count === 0);
    if (categoriesWithoutSubs.length === 0) {
        console.log('   ✅ All categories have subcategories');
    } else {
        console.log(`   ⚠️ ${categoriesWithoutSubs.length} categories without subcategories:`);
        categoriesWithoutSubs.forEach(cat => console.log(`      - ${cat.name}`));
    }
    
    // Subcategories without valid parent
    const orphanedSubcategories = subcategories.filter(sub => {
        const parent = categoriesCollection.findOne({ $loki: sub.categoryId });
        return !parent;
    });
    
    if (orphanedSubcategories.length === 0) {
        console.log('   ✅ All subcategories have valid parent categories');
    } else {
        console.log(`   ⚠️ ${orphanedSubcategories.length} orphaned subcategories found:`);
        orphanedSubcategories.forEach(sub => console.log(`      - ${sub.name} (categoryId: ${sub.categoryId})`));
    }
    
    // Active vs Inactive counts
    const activeCategories = categories.filter(cat => cat.isActive).length;
    const inactiveCategories = categories.length - activeCategories;
    const activeSubcategories = subcategories.filter(sub => sub.isActive).length;
    const inactiveSubcategories = subcategories.length - activeSubcategories;
    
    console.log('\n📊 ACTIVITY STATUS:');
    console.log(`   Categories: ${activeCategories} active, ${inactiveCategories} inactive`);
    console.log(`   Subcategories: ${activeSubcategories} active, ${inactiveSubcategories} inactive`);
    
    console.log('\n✅ DATABASE CHECK COMPLETE!');
});