const Loki = require('lokijs');

console.log('🧹 Cleaning Up Category Database...');

const db = new Loki('users.db', {
  autoload: true,
  autoloadCallback: () => {
    console.log('✅ Database loaded');
    
    const categories = db.getCollection('categories');
    const subcategories = db.getCollection('subcategories');
    
    console.log(`📋 Before cleanup - Categories: ${categories.data.length}, Subcategories: ${subcategories.data.length}`);
    
    // Find and remove duplicate categories
    const seenCategories = new Map();
    const duplicateCategories = [];
    
    categories.data.forEach(cat => {
      const key = cat.name.toLowerCase().trim();
      if (seenCategories.has(key)) {
        duplicateCategories.push(cat);
        console.log(`🔍 Found duplicate category: ${cat.name} (ID: ${cat.id})`);
      } else {
        seenCategories.set(key, cat);
      }
    });
    
    // Remove duplicate categories and their subcategories
    duplicateCategories.forEach(dupCat => {
      // Remove subcategories of duplicate category
      const dupSubcategories = subcategories.data.filter(sub => sub.categoryId === dupCat.id);
      dupSubcategories.forEach(dupSub => {
        subcategories.remove(dupSub);
        console.log(`  ↳ Removed duplicate subcategory: ${dupSub.name}`);
      });
      
      // Remove duplicate category
      categories.remove(dupCat);
      console.log(`❌ Removed duplicate category: ${dupCat.name}`);
    });
    
    // Clean up orphaned subcategories (subcategories without valid parent category)
    const validCategoryIds = new Set(categories.data.map(cat => cat.id));
    const orphanedSubcategories = subcategories.data.filter(sub => !validCategoryIds.has(sub.categoryId));
    
    orphanedSubcategories.forEach(orphan => {
      subcategories.remove(orphan);
      console.log(`🗑️  Removed orphaned subcategory: ${orphan.name}`);
    });
    
    // Remove duplicate subcategories within the same category
    const categorySubcategoryMap = new Map();
    const duplicateSubcategories = [];
    
    subcategories.data.forEach(sub => {
      const key = `${sub.categoryId}-${sub.name.toLowerCase().trim()}`;
      if (categorySubcategoryMap.has(key)) {
        duplicateSubcategories.push(sub);
        console.log(`🔍 Found duplicate subcategory: ${sub.name} in category ${sub.categoryName}`);
      } else {
        categorySubcategoryMap.set(key, sub);
      }
    });
    
    duplicateSubcategories.forEach(dupSub => {
      subcategories.remove(dupSub);
      console.log(`❌ Removed duplicate subcategory: ${dupSub.name}`);
    });
    
    // Update sort orders for categories
    categories.data
      .sort((a, b) => a.name.localeCompare(b.name))
      .forEach((cat, index) => {
        cat.sortOrder = index + 1;
      });
    
    // Update sort orders for subcategories within each category
    const categoriesMap = new Map();
    categories.data.forEach(cat => categoriesMap.set(cat.id, cat));
    
    subcategories.data.forEach(sub => {
      const parentCategory = categoriesMap.get(sub.categoryId);
      if (parentCategory) {
        sub.categoryName = parentCategory.name;
      }
    });
    
    // Group subcategories by category and update sort orders
    const subcategoriesByCategory = new Map();
    subcategories.data.forEach(sub => {
      if (!subcategoriesByCategory.has(sub.categoryId)) {
        subcategoriesByCategory.set(sub.categoryId, []);
      }
      subcategoriesByCategory.get(sub.categoryId).push(sub);
    });
    
    subcategoriesByCategory.forEach((subs, categoryId) => {
      subs.sort((a, b) => a.name.localeCompare(b.name))
          .forEach((sub, index) => {
            sub.sortOrder = index + 1;
          });
    });
    
    // Save the cleaned database
    db.saveDatabase();
    
    console.log('\\n🎉 Database Cleanup Complete!');
    console.log(`📊 After cleanup - Categories: ${categories.data.length}, Subcategories: ${subcategories.data.length}`);
    
    // Show final category list
    console.log('\\n📁 Final Category List:');
    categories.data
      .sort((a, b) => a.sortOrder - b.sortOrder)
      .forEach((cat, i) => {
        const subCount = subcategories.data.filter(sub => sub.categoryId === cat.id).length;
        console.log(`${String(i+1).padStart(2, ' ')}. ${cat.icon} ${cat.name} (${subCount} subcategories)`);
      });
    
    console.log('\\n✅ Your e-commerce platform database is now clean and optimized!');
    process.exit(0);
  }
});