const loki = require('lokijs');

// Initialize database
const db = new loki('users.db');

// Subcategories to add for categories that don't have any
const subcategoriesToAdd = {
    'Electronics': [
        { name: 'Smartphones', slug: 'smartphones', description: 'Mobile phones and accessories' },
        { name: 'Laptops', slug: 'laptops', description: 'Portable computers and laptops' },
        { name: 'Tablets', slug: 'tablets', description: 'Tablet devices and accessories' },
        { name: 'Smart Watches', slug: 'smart-watches', description: 'Wearable smart devices' },
        { name: 'Headphones', slug: 'headphones', description: 'Audio devices and headphones' },
        { name: 'Cameras', slug: 'cameras', description: 'Digital cameras and photography equipment' },
        { name: 'TVs & Audio', slug: 'tvs-audio', description: 'Television and audio systems' },
        { name: 'Gaming Consoles', slug: 'gaming-consoles', description: 'Video game consoles and accessories' }
    ],
    'Fashion': [
        { name: 'Men\'s Clothing', slug: 'mens-clothing', description: 'Apparel for men' },
        { name: 'Women\'s Clothing', slug: 'womens-clothing', description: 'Apparel for women' },
        { name: 'Men\'s Shoes', slug: 'mens-shoes', description: 'Footwear for men' },
        { name: 'Women\'s Shoes', slug: 'womens-shoes', description: 'Footwear for women' },
        { name: 'Bags & Purses', slug: 'bags-purses', description: 'Handbags, purses, and bags' },
        { name: 'Jewelry', slug: 'jewelry', description: 'Fashion jewelry and accessories' },
        { name: 'Watches', slug: 'watches', description: 'Fashion and luxury watches' },
        { name: 'Sunglasses', slug: 'sunglasses', description: 'Sunglasses and eyewear' }
    ],
    'Home & Garden': [
        { name: 'Living Room', slug: 'living-room', description: 'Living room furniture and decor' },
        { name: 'Bedroom', slug: 'bedroom', description: 'Bedroom furniture and accessories' },
        { name: 'Kitchen & Dining', slug: 'kitchen-dining', description: 'Kitchen and dining room items' },
        { name: 'Bathroom', slug: 'bathroom', description: 'Bathroom fixtures and accessories' },
        { name: 'Garden & Outdoor', slug: 'garden-outdoor', description: 'Outdoor and garden equipment' },
        { name: 'Home Decor', slug: 'home-decor', description: 'Decorative items for home' },
        { name: 'Storage & Organization', slug: 'storage-organization', description: 'Storage solutions and organizers' },
        { name: 'Lighting', slug: 'lighting', description: 'Indoor and outdoor lighting' }
    ],
    'Sports': [
        { name: 'Fitness Equipment', slug: 'fitness-equipment', description: 'Exercise and fitness gear' },
        { name: 'Team Sports', slug: 'team-sports', description: 'Equipment for team sports' },
        { name: 'Individual Sports', slug: 'individual-sports', description: 'Equipment for individual sports' },
        { name: 'Outdoor Recreation', slug: 'outdoor-recreation', description: 'Outdoor activity equipment' },
        { name: 'Athletic Wear', slug: 'athletic-wear', description: 'Sports clothing and apparel' },
        { name: 'Sports Accessories', slug: 'sports-accessories', description: 'Sports-related accessories' },
        { name: 'Water Sports', slug: 'water-sports', description: 'Equipment for water activities' },
        { name: 'Winter Sports', slug: 'winter-sports', description: 'Equipment for winter sports' }
    ]
};

db.loadDatabase({}, function() {
    console.log('Database loaded');
    
    // Get collections
    const categoriesCollection = db.getCollection('categories');
    const subcategoriesCollection = db.getCollection('subcategories');
    
    if (!categoriesCollection || !subcategoriesCollection) {
        console.log('Collections not found');
        return;
    }
    
    let addedCount = 0;
    
    // Process each category that needs subcategories
    Object.keys(subcategoriesToAdd).forEach(categoryName => {
        const category = categoriesCollection.findOne({ name: categoryName });
        if (!category) {
            console.log(`Category "${categoryName}" not found`);
            return;
        }
        
        console.log(`\nAdding subcategories for "${categoryName}" (ID: ${category.$loki}):`);
        
        const subcategoriesToCreate = subcategoriesToAdd[categoryName];
        subcategoriesToCreate.forEach(subcat => {
            // Check if subcategory already exists
            const existingSubcat = subcategoriesCollection.findOne({ 
                name: subcat.name, 
                categoryId: category.$loki 
            });
            
            if (existingSubcat) {
                console.log(`  - "${subcat.name}" already exists, skipping`);
                return;
            }
            
            const newSubcategory = {
                name: subcat.name,
                slug: subcat.slug,
                description: subcat.description,
                categoryId: category.$loki,
                isActive: true,
                createdAt: new Date().toISOString(),
                updatedAt: new Date().toISOString(),
                icon: '📦', // Default icon, can be customized later
                sortOrder: 0
            };
            
            subcategoriesCollection.insert(newSubcategory);
            console.log(`  ✅ Added "${subcat.name}"`);
            addedCount++;
        });
    });
    
    // Save the database
    db.saveDatabase(function(err) {
        if (err) {
            console.log('Error saving database:', err);
        } else {
            console.log(`\n✅ Successfully added ${addedCount} new subcategories!`);
            
            // Verify the results
            console.log('\n=== VERIFICATION ===');
            const totalSubcategories = subcategoriesCollection.count();
            console.log(`Total subcategories in database: ${totalSubcategories}`);
            
            // Check each category now has subcategories
            Object.keys(subcategoriesToAdd).forEach(categoryName => {
                const category = categoriesCollection.findOne({ name: categoryName });
                if (category) {
                    const count = subcategoriesCollection.count({ categoryId: category.$loki });
                    console.log(`${categoryName}: ${count} subcategories`);
                }
            });
        }
    });
});